from typing import Tuple, List
from threading import Thread
import logging

from http.server import BaseHTTPRequestHandler, HTTPServer

class ApiRequestHandler( BaseHTTPRequestHandler ):
    abclHandler   = None
    statusHandler = None
    
    def getRoute(self) -> str :
        """
        Get the route of the request, ie, excluding the
        query parameters
        """
        return self.path.split("?")[0]


    def getCommands(self) -> List[str] :
        """
        Get the ABCL commands from the query part of the
        request.
        
        Valid only for command.cgi
        """
        tmp         = self.path.split("?")
        if len(tmp[1])==0:
            return list()
        else:
            return tmp[1].split("&")


    def do_POST(self):
        logger          = logging.getLogger("http")
        route           = self.getRoute()
        if route == "/command.cgi" or route=="/basic.cgi":
            commands    = self.getCommands()
            if len(commands)==0 or len(commands)>1 :
                self.send_error(500,"Too many commands")
            else:
                self.send_response(200)
                self.wfile.write(ApiRequestHandler.abclHandler(commands[0]))
        else:
            self.send_error(404,"Not Found")

    def do_GET(self):
        logger          = logging.getLogger("http")
        route           = self.getRoute()
        if route=="/command.cgi" or route=="/basic.cgi":
            commands    = self.getCommands()
            if len(commands)==0 or len(commands)>1 :
                self.send_error(500,"Too many commands")
            else:
                logger.info("executing command: '{}'".format(commands[0]))
                reply   = ApiRequestHandler.abclHandler(commands[0])
                logger.info("got reply: {}".format(reply))
                self.send_response(200)
                self.end_headers()
                if reply is not None:
                    self.wfile.write(reply.encode("utf-8"))
        elif route=="/status":
            logger.info("retrieving epic app status")
            reply   = ApiRequestHandler.statusHandler()
            logger.info("got reply: {}".format(reply))
            self.send_response(200)
            self.end_headers()
            if reply is not None:
                    self.wfile.write(reply)
        else:
            self.send_error(404,"Not Found")


class HttpApi:
    def __init__(self,address:str,port:int):
        self.logger     = logging.getLogger("http-api")
        self.address    = address
        self.port       = port
        self.worker     = None

    def runServer(self):
        self.logger.info("creating the http server on {}:{} ...".format(self.address,self.port))
        self.server     = HTTPServer((self.address,self.port),ApiRequestHandler)
        self.logger.info("serving http...")
        self.server.serve_forever(poll_interval=0.5)
        self.logger.info("http server was shutdown. terminating worker thread.")


    def launch(self):
        if self.worker is not None:
            raise "Http server is already started"
        self.worker     = Thread(target=self.runServer)
        self.worker.daemon = True
        self.worker.start()

    def stop(self):
        if self.worker is None:
            pass
        self.logger.info("stopping http server...")
        self.server.shutdown()
        self.logger.info("waiting for worker thread...")
        self.worker.join()
        self.logger.info("http API is now stopped.")
        self.worker = None

    def setAbclHandler(self,handler):
        ApiRequestHandler.abclHandler   = handler

    def setStatusHandler(self,handler):
        ApiRequestHandler.statusHandler = handler



