#!/usr/bin/env python3

from os import statvfs_result
import threading
from types import ClassMethodDescriptorType
from urllib.parse import urlparse
import time
import signal
import logging
from logging.handlers import RotatingFileHandler
import syslog

from serial.serialposix import Serial
import alsaaudio
import sys
from uci import Uci
from typing import *
import random
import json
import traceback
import subprocess

from barix.media.RtpAudioSource import *
from barix.media.SipClientAudioSource import *
from barix.io.Gpio import Gpio
from barix.io import leds
from barix.licensing import *
from barix.web.system_info import getMACaddr
import baco

from http_api import HttpApi
from playback_manager import *
from fw_updater import FwUpdater
from tcp_api import TcpApi
from serial_gw import SerialGw

class InOut:
    IDLE     = 0
    PRESSED  = 1
    RELEASED = 2

    def __init__(self, pin:int ):
        self.lock   = threading.Lock()
        self.mode   = Gpio.IN
        self.outVal = None
        self.gpio   = Gpio(str(pin),"in")
        self.state  = self.gpio.read()
        self.relayOn= False


    def checkEvents(self) -> int:
        """
        Check for updates on the input
        """
        try:
            self.lock.acquire()
            if self.relayOn:
                return InOut.IDLE

            stt = self.gpio.read()
            if self.state==stt:
                return InOut.IDLE

            self.state  = stt
            if stt==1:
                return InOut.RELEASED
            return InOut.PRESSED
        finally:
            self.lock.release()


    def read(self) -> int:
        """
        Read the input. Does not update event tracking.
        """
        try:
            self.lock.acquire()
            if self.relayOn:
                return 1

            return self.gpio.read()
        finally:
            self.lock.release()


    def setRelay(self,value: bool) -> None:
        try:
            self.lock.acquire()
            if value:
                if not self.relayOn:
                    self.gpio.setDirection(Gpio.OUT)
                    self.gpio.write(0)
                    self.relayOn    = True
            else:
                self.gpio.setDirection(Gpio.IN)
                self.relayOn    = False
        finally:
            self.lock.release()

    def getMode(self) -> str:
        if self.relayOn:
            return "RELAY"
        return "INPUT"

def uciGet( uci: Uci, config: str, section: str, opt: str) -> str:
    try:
        return uci.get(config,section,opt)
    except:
        return None


def decodeDCommand(line:str) -> Tuple[str,str]:
    if len(line)<2:
        return ("",None)

    if line=="DS":
        return ("DS",None)
    elif line=="DA":
        return ("DA",None)
    else:
        # Dsip://id@server
        # DHext
        # DHsip://id@server
        # Dext
        if line.find("DH")==0:
            if len(line)>3:
                return ("DH",line[2:])
            else:
                # DH without arguments makes no sense!!
                return ("",None)
        else:
            return ( "D" , line[1:])


class EpicSip:
    HTTP_API_PORT   = 48081

    sipDict = {
        SipCallState.IDLE:              "IDLE",
        SipCallState.CALLING:           "CALLING",
        SipCallState.INCOMMING_CALL:    "INCOMMING",
        SipCallState.RINGING:           "RINGING",
        SipCallState.CALLING_RINGING:   "RINGING",
        SipCallState.CONNECTED:         "CONNECTED",
        SipCallState.GETTING_NOTIF:     "GETTING_NOTIF"
    }

    def __init__(self):
        self.logger                 = logging.getLogger("epic")
        with open("/barix/info/VERSION","r") as fIn:
            self.sysVersion = fIn.readline().strip()

        with open("/barix/apps/epic-sip/MODEL") as fIn:
            m               = fIn.readline().strip()
            if m=="MS-300B":
                self.model  = "MS-300B"
            else:
                self.model  = "MS-500A"

        syslog.openlog("AE-MSx00")
        msg = "MS-x00 controller starting. model={} sysVersion={}".format(self.model, self.sysVersion)
        self.logger.info(msg)
        syslog.syslog(msg)
        self.stopApplication        = False
        signal.signal(signal.SIGTERM,self.signalHandler)
        signal.signal(signal.SIGINT,self.signalHandler)


    def setRelay1(self,stt:bool) -> None:
        """
        Sets the state of relay1 to ON(True) or OFF(False)

        Returns true is state changed, which requires a status
        update to be sent through TCP connected clients
        """
        if stt:
            sttStr = "ON"
        else:
            sttStr = "OFF"

        syslog.syslog("RELAY1 --> {}".format(sttStr))
        self.io1.setRelay(stt)


    def setRelay2(self,stt:bool) -> None:
        """
        Sets the state of relay2 to ON(True) or OFF(False)

        Returns true is state changed, which requires a status
        update to be sent through TCP connected clients
        """
        if stt:
            sttStr = "ON"
        else:
            sttStr = "OFF"

        syslog.syslog("RELAY2 --> {}".format(sttStr))
        self.io2.setRelay(stt)


    def signalHandler(self, sigNum: int, frame ):
        msg     = "Cauhght signal {}. signaling shutdown".format(sigNum)
        print(msg)
        self.logger.info(msg)
        self.stopApplication    = True

    def bringUpMixers(self):
        self.logger.info("bringing up mixer channels")
        alsaaudio.PCM(device='channel1')
        alsaaudio.PCM(device='channel2')
        alsaaudio.PCM(device='channel3')
        alsaaudio.PCM(device='channel4')
        alsaaudio.PCM(device='channel5')

    def setMixerParam(self, param, value):
        p = subprocess.Popen("amixer -q -c 2 -- sset "+"\'"+param+"\'"+" \'"+value+"\'", shell=True)
        p.wait()

    def setAudioInputSource(self):
        logMsg  = "Audio input: {}".format(self.audioInput)
        self.logger.info(logMsg)
        syslog.syslog(logMsg)
        if self.audioInput == "LINEIN":
            self.setMixerParam("Line In","cap")
            self.setMixerParam("ADC Gain","0dB")
            self.setMixerParam("Mic1","nocap")
        elif self.audioInput == "MICIN":
            self.setMixerParam("Mic1","cap")
            self.setMixerParam("Mic1","4.5dB")
            self.setMixerParam("Mic1 Boost",str(str(self.micInGain) + "dB"))
            self.setMixerParam("Line In","nocap")
        else:
            self.logger.error("Invalid audio input")

    def run(self):
        try:
            uci             = Uci()
            fwUrl           = uciGet(uci,"epic","main","fw_update_url")
            fwAutoUpdate    = uciGet(uci,"epic","main","fw_update_auto")
            if fwUrl is not None and fwAutoUpdate=="true":
                self.fwUpdater  = FwUpdater(fwUrl,"/mnt/data/state/epic-sip/fw_last_modified.conf")
                self.fwUpdater.setIniWindowTime(900)
                self.fwUpdater.setUpdateBaseTime(2,0)
                self.fwUpdater.setUpdateWindowDuration(120)
            else:
                self.fwUpdater  = None

            self.io1            = InOut(10)
            self.io2            = InOut(13)
            # initially both relays OFF
            self.io1.setRelay(False)
            self.io2.setRelay(False)
            self.redLed         = leds.Led("barix:led1:red")
            self.greenLed       = leds.Led("barix:led1:green")
            self.greenLed.set(True)
            self.redLed.set(False)
            syslog.syslog("RED LED --> OFF")
            syslog.syslog("GREEN LED --> ON")

            self.mainVolume = int(uci.get("epic","main","volume"))
            if self.mainVolume<0:
                self.mainVolume = 0
            elif self.mainVolume>100:
                self.mainVolume = 100

            self.mainVolumeCtl  = alsaaudio.Mixer("BARIX_MAIN")
            syslog.syslog("MAIN VOLUME --> {}%".format(self.mainVolume))
            self.mainVolumeCtl.setvolume(self.mainVolume)
            self.bringUpMixers()
            self.httpApi                = HttpApi("0.0.0.0",EpicSip.HTTP_API_PORT)
            self.httpApi.setAbclHandler(self.executeAbcl)
            self.httpApi.setStatusHandler(self.getAppStatus)
            self.httpApi.launch()


            tcpApiPort          = int(uciGet(uci,"epic","api","abcl_tcp_port"))
            if tcpApiPort!=0:
                self.tcpApi                 = TcpApi("0.0.0.0",tcpApiPort)
                self.tcpApi.setAbclHandler(self.executeAbcl)
                self.tcpApi.launch()
            else:
                self.tcpApi     = None

            self.playbackManager        = PlaybackManager()

            bgmUrl  = uciGet(uci,"epic","bgm","url")
            bgmBufSz= int(uci.get("epic","bgm","buffer_size"))
            bgmVol  = int(uci.get("epic","bgm","volume"))
            if bgmUrl is not None and len(bgmUrl)>0:
                urlComps    = urlparse(bgmUrl)
                if urlComps.scheme=="rtp":
                    bgmParams           = dict()
                    bgmParams["url"]    = bgmUrl
                    bgmParams["playback_mode"] = "full_buffer"
                    bgmParams["input_buffer_size"] = 0
                    bgmParams["output_buffer_size"] = bgmBufSz
                    bgmSrc  = RtpAudioSource(1,"bgm",bgmVol,bgmParams,"chan_1", 48000)
                    logMsg  = "BGM SOURCE: {} buff={}ms vol={}%".format(bgmUrl,bgmBufSz,bgmVol)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.playbackManager.setBgmSource(bgmSrc)
                else:
                    self.logger.error("BGM url is not supported: {}".format(bgmUrl))

            hpnUrl  = uciGet(uci,"epic","hpn","url")

            if hpnUrl is not None and len(hpnUrl)>0:
                hpnBufSz= int(uci.get("epic","hpn","buffer_size"))
                hpnVol  = int(uci.get("epic","hpn","volume"))
                if uci.get("epic","hpn","high_prio").lower()=="true":
                    self.hpnHighPrio    = True
                else:
                    self.hpnHighPrio    = False

                urlComps    = urlparse(hpnUrl)
                if urlComps.scheme=="rtp":
                    hpnParams           = dict()
                    hpnParams["url"]    = hpnUrl
                    hpnParams["playback_mode"] = "low_latency"
                    hpnParams["input_buffer_size"] = 0
                    hpnParams["output_buffer_size"] = hpnBufSz
                    hpnSrc  = RtpAudioSource(2,"hpn",hpnVol,hpnParams,"chan_2", 48000)
                    self.playbackManager.setHpnSource(hpnSrc)
                    logMsg  = "HPN SOURCE: {} buff={}ms vol={}%".format(hpnUrl,hpnBufSz,hpnVol)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.playbackManager.setHpnHighPrio(self.hpnHighPrio)
                else:
                    self.logger.error("HPN url is not supported: {}".format(hpnUrl))

            self.autodial1  = uciGet(uci,"epic","sip","autodial1")
            self.autodial2  = uciGet(uci,"epic","sip","autodial2")

            sipEnable           = uciGet(uci,"epic","sip","enabled")
            self.audioInput     = uciGet(uci,"epic","sip","input")
            self.micInGain      = uciGet(uci,"epic","sip","micin_gain")
            self.lineInGain     = uciGet(uci,"epic","sip","linein_gain")
            sipVol              = int(uci.get("epic","sip","volume"))
            sipDomain           = uciGet(uci,"epic","sip","domain")
            self.sipUserName    = uciGet(uci,"epic","sip","username")
            sipPassword         = uciGet(uci,"epic","sip","password")
            autoAnswer          = uciGet(uci,"epic","sip","auto_answer")
            answerTime          = uciGet(uci,"epic","sip","answer_time")
            callTimeout         = uciGet(uci,"epic","sip","call_timeout")
            streamTimeout       = uciGet(uci,"epic","sip","stream_timeout")
            sipTransport        = uciGet(uci,"epic","sip","transport")
            beepOnAnswer        = uciGet(uci,"epic","sip","beep_on_answer")
            halfDuplex          = uciGet(uci,"epic","sip","half_duplex")
            aecEnabled          = uciGet(uci,"epic","sip","aec")
            inputGain           = 0.01*int(uciGet(uci,"epic","sip","input_gain"))
            hdxLevel            = float(uciGet(uci,"epic","sip","hdx_level"))
            hdxTimeout          = 0.001*float(uciGet(uci,"epic","sip","hdx_timeout"))

            self.setAudioInputSource()

            self.sipSrc = None
            if sipEnable=="true":
                if sipDomain is not None and self.sipUserName is not None and sipPassword is not None:
                    sipParams                               = dict()
                    sipParams["sip_mode"]                   = "server"
                    sipParams["domain"]                     = sipDomain
                    sipParams["user_name"]                  = self.sipUserName
                    sipParams["password"]                   = sipPassword
                    sipParams["transport_and_encryption"]   = sipTransport
                    sipParams["auto_answer"]                = autoAnswer
                    sipParams["answer_time"]                = answerTime
                    sipParams["call_timeout"]               = callTimeout
                    sipParams["stream_timeout"]             = streamTimeout
                    sipParams["beep_on_answer"]             = beepOnAnswer
                    sipParams["half_duplex"]                = halfDuplex
                    sipParams["aec"]                        = aecEnabled
                    sipParams["input_gain"]                 = inputGain
                    sipParams["hdx_level"]                  = hdxLevel
                    sipParams["hdx_timeout"]                = hdxTimeout
                    self.logger.info("loading SIP with '{}'".format(sipParams))
                    self.sipSrc = SipClient(3,"sip",sipVol,sipParams,"chan_3","plug:dsnoop_analog")
                    self.playbackManager.setSipSource(self.sipSrc)
                    syslog.syslog("Loaded SIP: domain='{}', user='{}'".format(sipDomain,self.sipUserName))
                else:
                    self.logger.error("SIP is enabled but some parameters are missing!")

            self.serialGw   = None
            serialGwMode    = uciGet(uci,"epic","serialgw","mode")
            serialBaud      = int(uciGet(uci,"epic","serialgw","baud"))
            serialDatabits  = int(uciGet(uci,"epic","serialgw","databits"))
            serialParity    = uciGet(uci,"epic","serialgw","parity")
            serialStopBits  = int(uciGet(uci,"epic","serialgw","stopbits"))
            serialPeerPort  = int(uciGet(uci,"epic","serialgw","port"))
            serialPeerPort  = int(uciGet(uci,"epic","serialgw","port"))
            serialPeerHost  = uciGet(uci,"epic","serialgw","host")
            if serialGwMode=="server":
                self.serialGw   = SerialGw("/dev/ttyS1",SerialGw.SERVER)
            elif serialGwMode=="client":
                self.serialGw   = SerialGw("/dev/ttyS1",SerialGw.CLIENT)
            elif serialGwMode=="disabled":
                self.serialGw   = SerialGw("/dev/ttyS1",SerialGw.NOGW)
            else:
                self.logger.error("unknown serialgw mode: {}".format(serialGwMode))

            self.serialGw.setSerialParams(serialBaud,serialDatabits,serialParity,serialStopBits)
            self.serialGw.setPeer(serialPeerPort,serialPeerHost)
            self.serialGw.start()
            logMsg      = "Loaded serial to TCP gateway: mode='{}'".format(serialGwMode)
            self.logger.info(logMsg)
            syslog.syslog(logMsg)


            self.playingNotification    = False
            self.notifyRelay1           = False
            self.notifyRelay2           = False
            if uciGet(uci,"epic","io","notify_relay1")=="true":
                self.notifyRelay1       = True
            if uciGet(uci,"epic","io","notify_relay2")=="true":
                self.notifyRelay2       = True

            logMsg      = "Initialization is complete! Entering main Loop..."
            self.logger.info(logMsg)
            syslog.syslog(logMsg)
            self.runMainLoop()
            sys.exit(0)
        except:
            self.logger.error("main loop finished with exception")
            traceback.print_exc()
            sys.exit(1)

    def buildDAE(self) -> str:
        with open("/sys/class/thermal/thermal_zone0/temp") as f:
            temp = int(f.readlines()[0].strip())
            temp = int(temp/1000)
        with open("/proc/uptime") as f:
            tmp=f.readlines()[0].strip()
            tmp=tmp.split(" ")
            uptime = int(float(tmp[0]))
            utDays = int(uptime/86400)
            utHours= int((uptime-utDays*86400)/3600)
            utMinutes=int((uptime-utDays*86400-utHours*3600)/60)
            uptime = "{:02d}{:02d}{:02d}".format(utDays, utHours, utMinutes)
        with open("/proc/meminfo") as f:
            tmp=f.readlines()[1].strip()
            tmp=tmp.split(":")[1].strip()
            tmp=tmp.split(" ")[0]
            mem = int(tmp)
            mem = int(mem/1024)
        with open("/sys/class/net/eth0/carrier_changes") as f:
            lnk_chng = int(f.readlines()[0].strip()) - 2
        message="$DAE:{}:{}:{}:{}\r\n"
        if temp>90:
            message="!!!!!!!!!!!!!!!$DAE:{}:{}:{}:{}!!!!!!!!!!!!!!!\r\n"
        elif temp>80:
            message=">>>>>>>>>>>>>>>$DAE:{}:{}:{}:{}<<<<<<<<<<<<<<<\r\n"
        elif temp>75:
            message="###############$DAE:{}:{}:{}:{}###############\r\n"
        message=message.format(temp, uptime, mem, lnk_chng)
        return message

    def runMainLoop(self):
        loopCount   = 0
        lastSipState= None
        while not self.stopApplication:
            statusChanged   = False

            # iterate fw update checks at ~12,5 seconds
            if self.fwUpdater is not None and loopCount%50==0:
                self.fwUpdater.checkFwUpdates()

            evt     = self.io1.checkEvents()
            if evt==InOut.PRESSED:
                if self.sipSrc is not None:
                    # if SIP is IDLE, dial self.autodial1 (if not None)
                    # answer if RINGING
                    # hangup if a call is active
                    currentSipState = self.sipSrc.reportSipCallState()
                    if currentSipState == SipCallState.IDLE and self.autodial1 is not None:
                        logMsg  = "SIP AUTODIAL 1: calling {}".format(self.autodial1)
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.makeCall(self.autodial1)

                    if currentSipState == SipCallState.RINGING and self.sipSrc is not None:
                        logMsg  = "SIP RINGING + INPUT1 --> ANSWER"
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.answerCall()

                    if currentSipState != SipCallState.IDLE and currentSipState != SipCallState.RINGING and self.sipSrc is not None:
                        logMsg  = "SIP is not IDLE and not receiving a call + INPUT1 --> HANGUP"
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.hangupCall()

                statusChanged   = True
            elif evt==InOut.RELEASED:
                statusChanged   = True

            evt     = self.io2.checkEvents()
            if evt==InOut.PRESSED:
                statusChanged   = True
                # if SIP is IDLE, dial self.autodial2 (if not None)
                # NB: does not answer if ringing
                if not self.sipSrc == None:
                    currentSipState = self.sipSrc.reportSipCallState()
                    if currentSipState == SipCallState.IDLE and self.autodial2 is not None and self.sipSrc is not None:
                        logMsg  = "SIP AUTODIAL 2: calling {}".format(self.autodial2)
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.makeCall(self.autodial2)

            elif evt==InOut.RELEASED:
                statusChanged   = True

            notificationState   = self.playbackManager.hpnIsActive() or self.playbackManager.sipIsActive()
            if self.playingNotification!=notificationState:
                #statusChanged   = True
                if notificationState:
                    stateStr    = "ON"
                    logMsg      = "Notify DSP, RED LED --> ON"
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.serialGw.writeSideChannel("$SXI:6\r\n")

                else:
                    stateStr    = "OFF"
                    logMsg      = "Notify DSP, RED LED --> OFF"
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.serialGw.writeSideChannel("$SXI:0\r\n")

                self.redLed.set(notificationState)

                if self.notifyRelay1:
                    logMsg      = "RELAY1 --> {}".format(stateStr)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.io1.setRelay(notificationState)

                if self.notifyRelay2:
                    logMsg      = "RELAY2 --> {}".format(stateStr)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.io2.setRelay(notificationState)

            self.playingNotification    = notificationState


            if self.sipSrc is not None:
                newSipState     = self.sipSrc.reportStatus()
                sipCallState    = newSipState["callState"]
                if lastSipState is None or sipCallState!=lastSipState["callState"]:
                    #statusChanged=True
                    stateStr    = "????"
                    if  sipCallState==SipCallState.IDLE:
                        stateStr    = "IDLE"
                    elif sipCallState==SipCallState.CALLING:
                        stateStr    = "CALLING dest='{}'".format(newSipState["currentCall"])
                    elif sipCallState==SipCallState.INCOMMING_CALL:
                        stateStr    = "INCOMMING"
                    elif sipCallState==SipCallState.RINGING or sipCallState==SipCallState.CALLING_RINGING:
                        stateStr    = "RINGING peer='{}'".format(newSipState["currentCall"])
                    elif sipCallState==SipCallState.CONNECTED:
                        stateStr    = "CONNECTED peer='{}'".format(newSipState["currentCall"])
                    elif sipCallState==SipCallState.GETTING_NOTIF:
                        stateStr    = "NOTIFICATION"
                    logMsg          = "SIP Call State --> {}".format(stateStr)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)

                registered          = newSipState["registered"]
                serverRespCode      = newSipState["sipResponseCode"]
                if lastSipState is None or registered!=lastSipState["registered"] or serverRespCode!=lastSipState["sipResponseCode"]:
                    logMsg          = "SIP Registration: {}, {}".format(registered.upper(), serverRespCode)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                lastSipState        = newSipState

            if statusChanged:
                status          = self.getDeviceStatus()
                self.logger.info("broadcasting status update: {}".format(status))
                self.tcpApi.broadcastStatusUpdate(status)

            loopCount   += 1
            time.sleep(0.1)

        self.logger.info("shutting down MS-x00...")
        if self.serialGw is not None:
            self.logger.info("shutting down serial gw...")
            self.serialGw.shutdown()
        self.logger.info("shutting down playback manager...")
        self.playbackManager.shutdown()
        self.logger.info("shutting down http API...")
        self.httpApi.stop()
        self.logger.info("shutting down TCP api...")
        self.tcpApi.shutdown()
        sys.exit(0)

    def executeAbcl(self,line:str) -> str:
        if len(line)==0:
            return ""

        self.logger.info("ABCL command: cmd='{}'".format(line))
        cmd0    = line[0]

        if cmd0=="T":
            cmd=line[0:2]
            if cmd=="TC":
                with open("/sys/class/thermal/thermal_zone0/temp") as f:
                    t = int(f.readlines()[0].strip())
                    t=str(int(t/1000))
                    return t
            else:
                # all other arguments are ignored. might validate empty params...
                reply   = dict()
                reply["m"] = self.model
                reply["s"] = "IPAM-400"
                reply["v"] = self.sysVersion
                reply["a"] = getMACaddr()
                return json.dumps(reply,indent=4)
        elif cmd0=="D":
            (cmd,arg) = decodeDCommand(line)
            sipState = self.sipSrc.reportSipCallState()
            if cmd=="D":
                # dial SIP <arg> if SIP is IDLE
                if sipState == SipCallState.IDLE and self.sipSrc is not None:
                    logMsg  = "ABCL Dial (D) --> dial '{}'".format(arg)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.sipSrc.makeCall(arg)
            elif cmd=="DS":
                # Hangup any currently active SIP call
                if self.sipSrc is not None:
                    logMsg  = "ABCL Hangup (DS) --> hangup SIP call (if any)"
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.sipSrc.hangupCall()
            elif cmd=="DA":
                # answer the incoming SIP call
                if self.sipSrc is not None:
                    logMsg  = "ABCL Answer (DA) --> Answer SIP call (if any)"
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.sipSrc.answerCall()
            elif cmd=="DH":
                # dial SIP <arg>
                # if a call is active, terminate it and dial the new one
                if self.sipSrc is not None:
                    if sipState != SipCallState.IDLE:
                        logMsg  = "ABCL Dial-Prio (DH) --> hangup current call"
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.hangupCall()
                    logMsg  = "ABCL Dial-Prio (DH) --> dial '{}'".format(arg)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.sipSrc.makeCall(arg)
            else:
                self.logger.warning("unrecognized ABCL D*** command: '{}'".format(line))
        elif line.find("VS=")==0:
            if len(line)<4:
                self.logger.warning("malformed VS command: '{}'".format(line))
            else:
                vol     = int(line[3:])
                if vol<0 or vol>20:
                    self.logger.warning("bad volume in VS command: '{}' (expected=0..20)".format(vol))
                    return ""
                self.mainVolume = vol*5
                self.mainVolumeCtl.setvolume(self.mainVolume)
                logMsg  = "ABCL Set Volume (VS) --> volume={} / {}%".format(vol,self.mainVolume)
                self.logger.info(logMsg)
                syslog.syslog(logMsg)
                uci     = Uci()
                uci.set("epic","main","volume",str(self.mainVolume))
                uci.commit("epic")
        elif line == "VQ":
            return str(self.mainVolume//5)
        elif line == "Q":
            msg = self.getDeviceStatus()
            msg += "\r\n"
            msg += self.buildDAE()
            return msg
        elif line.find("R1=")==0 or line.find("R2=")==0:
            if len(line)<4:
                self.logger.warning("malformed R1/R2 command: '{}'".format(line))
                return ""

            relayNum  = int(line[1])
            intVal    = int(line[3])
            if intVal==1:
                relayVal = True
            else:
                relayVal = False

            if relayNum==1:
                self.setRelay1(relayVal)
            else:
                self.setRelay2(relayVal)
        else:
            self.logger.warning("unrecognized ABCL command: '{}'".format(line))

        return ""


    def getDeviceStatus(self) -> str:
        """
        * "SIP_ID": sip user id
        * STT:
            - S0 --> IDLE
            - S1 --> CALLING
            - S2 --> INCOMING CALL / RINGING
            - S3 --> CONNECTED
            - S4 --> GETTING NOTIFICATION??????
        """
        sipUid      = "0"   #invalid SIP ID
        sipState    = "S0"

        if self.sipSrc is not None:
            sipUid = self.sipUserName
            state = self.sipSrc.reportSipCallState()
            if state == SipCallState.IDLE:
                sipState = "S0"
            elif state == SipCallState.CALLING:
                sipState = "S1"
            elif state == SipCallState.INCOMMING_CALL or state == SipCallState.RINGING or state == SipCallState.CALLING_RINGING:
                sipState = "S2"
            elif state == SipCallState.CONNECTED:
                sipState = "S3"
            elif state == SipCallState.GETTING_NOTIF:
                sipState = "S4"

        ii          = 0x4
        if self.io1.read()!=0:
            ii     |= 0x1
        if self.io2.read()!=0:
            ii     |= 0x2

        oo          = 0
        if self.io1.getMode()=="RELAY":
            oo     |= 0x1
        if self.io2.getMode()=="RELAY":
            oo     |= 0x02
        return "{},{},{:02X},{:02X}".format(sipUid,sipState,ii,oo)


    def getAppStatus(self):

        audioStatus             = "IDLE"

        sipStatus               = "NOT CONFIGURED"
        sipRegStatus            = None
        sipRegistrationCode     = None
        sipPeer                 = None
        if self.sipSrc is not None:
            currSipStatus = self.sipSrc.reportStatus()
            sipRegStatus = str(currSipStatus["registered"]).upper()
            sipStatus = self.sipDict[currSipStatus["callState"]]
            sipRegistrationCode = currSipStatus["sipResponseCode"]
            sipPeer = currSipStatus["currentCall"]
            if self.playbackManager.sipIsActive():
                audioStatus = "SIP"

        hpnStatus = "NOT CONFIGURED"
        if self.playbackManager.hpnSource is not None:
            if self.playbackManager.hpnIsActive():
                hpnStatus = "ALIVE"
                audioStatus = "HPN"
            else:
                (state, alive) = self.playbackManager.hpnSource.getState()
                if alive:
                    hpnStatus = "ALIVE"
                else:
                    hpnStatus = "IDLE"

        bgmStatus = "NOT CONFIGURED"
        if self.playbackManager.bgmSource is not None:
            if self.playbackManager.bgmIsActive():
                bgmStatus = "ALIVE"
                audioStatus = "BGM"
            else:
                (state, alive) = self.playbackManager.bgmSource.getState()
                if alive:
                    bgmStatus = "ALIVE"
                else:
                    bgmStatus = "IDLE"

        ioStatus        = dict()
        ioStatus["io1"] = dict()
        if self.io1.getMode()=="RELAY":
            ioStatus["io1"]["mode"] = "RELAY"
            ioStatus["io1"]["state"]= "ON"
        else:
            ioStatus["io1"]["mode"] = "INPUT"
            if self.io1.read()==0:
                ioStatus["io1"]["state"] = "ON"
            else:
                ioStatus["io1"]["state"] = "OFF"


        ioStatus["io2"] = dict()
        if self.io2.getMode()=="RELAY":
            ioStatus["io2"]["mode"] = "RELAY"
            ioStatus["io2"]["state"]= "ON"
        else:
            ioStatus["io2"]["mode"] = "INPUT"
            if self.io2.read()==0:
                ioStatus["io2"]["state"] = "ON"
            else:
                ioStatus["io2"]["state"] = "OFF"


        status = {
                "sipStatus" : "{}".format(sipStatus),
                "sipRegStatus": "{}".format(sipRegStatus),
                "sipRegistrationCode" : "{}".format(sipRegistrationCode),
                "sipPeer" : "{}".format(sipPeer),
                "hpnStatus" : "{}".format(hpnStatus),
                "bgmStatus": "{}".format(bgmStatus),
                "audioStatus": "{}".format(audioStatus) ,
                "ioStatus" : ioStatus
                }
        status = json.dumps(status).encode('utf-8')
        return status

if __name__ == "__main__":
    rotHandler = RotatingFileHandler("/var/log/epic.log", maxBytes=2*1024*1024, backupCount=3)
    logging.basicConfig(level=logging.DEBUG,handlers=[rotHandler],format='%(asctime)s %(name)s %(levelname)s %(message)s')
    baco.initializeLogger("/var/log/baco.log",baco.LogPriority.INFO,2000000,5,False)
    found,_ = findLicense("core-image-ae-msx00")
    if not found:
        found,_ = findLicense("barix-development-platform")

    if not found:
        l = logging.getLogger("main")
        l.error("required license is missing: 'core-image-ae-msx00|barix-development-platform'. Exit.")
        sys.exit(1)

    baco.initializeAudioLib()
    app     = EpicSip()
    app.run()
    print("ALL FINISHED")
