import logging
import syslog
from os import times
import socket
from select import select
import threading
import time

class TcpApi:
    sktCon = None
    activeIP = None

    def __init__(self, host:str, port:int):
        self.logger = logging.getLogger("tcp")
        self.lock   = threading.Lock()

        self.peerHost   = host
        self.peerPort   = port
        self.running    = False
        self.stopApi     = False

        with open('EPIC_IP', 'w') as f:
            f.write('')
    
    
    def handleConnection(self, sktCon: socket.socket, stop):
        sktCon.settimeout(0.01)
        sktCon.setsockopt(socket.SOL_SOCKET,socket.SO_KEEPALIVE,1)
        sktCon.setsockopt(socket.IPPROTO_TCP, socket.TCP_KEEPIDLE, 15)
        sktCon.setsockopt(socket.IPPROTO_TCP, socket.TCP_KEEPINTVL, 2)
        sktCon.setsockopt(socket.IPPROTO_TCP, socket.TCP_KEEPCNT, 5)
        while not stop():
            # read from sktCon
            # # @catch socket.timeout --> pass
            # # @catch any other exception return (don't close TTY!!)
#            rdlist, _, _ = select((sktCon,),(),(),0.1)
            rdlist=select([sktCon], [], [], 0.1)[0]
            if rdlist:
                try:
                    sktData = sktCon.recv(100)
                    # print(sktData.decode())
                    if len(sktData)==0:
                        logMsg  = "ABCL TCP: lost connection to client unexpextedly--> cleanup"
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        sktCon.close()
                        TcpApi.activeIP = None
                        return
                    else:
                        try:
                            self.lock.acquire()
                            buff = sktData.decode().strip().split("\r\n")
                            reply = ""
                            for cmd in buff:
                                if cmd == "":
                                    continue
                                self.logger.info("From tcp: "+cmd)
                                r = self.abclHandler(cmd)
                                if len(r)>0:
                                    reply += r.strip()+"\r\n"
                            if len(reply)>0:
                                reply = reply + "\r\n"
                                # print(reply)
                                self.logger.debug("sending reply: {}".format(reply))
                                sktCon.sendall(reply.encode())
                                self.logger.debug("reply sent!")
                        except Exception as ex:
                            self.logger.debug(str(ex))
                        finally:
                            self.lock.release()
                except socket.timeout:
                    pass
                except Exception as ex:
                    logMsg  = "ABCL TCP: lost connection to client --> cleanup ({})".format(str(ex))
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    sktCon.close()
                    TcpApi.activeIP = None
                    return

        self.logger.info("ABCL TCP: Closing active connection")
        sktCon.close()
        TcpApi.activeIP = None


    def runServer(self):
        logMsg  = "ABCL TCP: listening on port {}".format(self.peerPort)
        self.logger.info(logMsg)
        syslog.syslog(logMsg)
        self.skt = socket.socket(socket.AF_INET,socket.SOCK_STREAM)
        self.skt.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR,1)
        # skt.setsockopt(socket.SOL_SOCKET, socket.SO_KEEPALIVE, 1)
        self.skt.bind(("0.0.0.0",self.peerPort))
        self.skt.listen(1)
        self.skt.settimeout(30)

        thrd = None
        while not self.stopApi:
            try:
                (tmpCon, clientAddr)  = self.skt.accept()

                # If already active connection reject attempts from other IPs
                if not TcpApi.activeIP == None and clientAddr[0] != TcpApi.activeIP:
                    tmpCon.close()
                    self.logger.info("ABCL TCP: connection from {} rejected".format(clientAddr[0]))
                    continue

                if not thrd == None:
                    stopHandling = True
                    thrd.join()
                    thrd = None

                self.sktCon=tmpCon
                TcpApi.activeIP=clientAddr[0]
                stopHandling = False

                with open('EPIC_IP', 'w') as f:
                    f.write(clientAddr[0])
                    
                logMsg = "ABCL TCP: client connected from {}".format(clientAddr[0])
                self.logger.info(logMsg)
                syslog.syslog(logMsg)

#                self.handleConnection(sktCon)
                thrd = threading.Thread(target=self.handleConnection, args=(self.sktCon, lambda: stopHandling))
                thrd.start()
            except socket.timeout:
                pass
            except Exception as ex:
                self.logger.error("Unexpected exception in listening socket: {}".format(ex))
            finally:
                pass

        self.logger.info("TcpApi server exiting.")


    def broadcastStatusUpdate(self, update:str) -> None:
        try:
            update = update + "\r\n"
            self.logger.debug("sending update: {}".format(update))
            self.sktCon.sendall(update.encode())
            self.logger.debug("update sent!")
        except Exception as ex:
            self.logger.debug(str(ex))
        

    def launch(self):
        """
        Start operation of the serial GW
        Must be called after set and setPeer if
        mode is different from MODE_DISABLED
        """
        try:
            self.lock.acquire()
            self.workerThread   = threading.Thread(target=self.runServer)
            self.workerThread.daemon    = True
            self.workerThread.start()

            self.running                = True
        finally:
            self.lock.release()

    def setAbclHandler(self, handler):
        self.abclHandler = handler


    def shutdown(self):
        """
        Shutdown operation of the serial GW and disconnect
        from the serial port
        """
        joinWorker  = False
        try:
            self.lock.acquire()
            if not self.running:
                return

            if self.workerThread is not None:
                joinWorker  = True
                self.stopApi = True

            self.running    = False
        finally:
            self.lock.release()

        if joinWorker:
            self.logger.info("waiting for worker thread to stop")
            self.workerThread.join()


