from os import listdir, path
from os.path import isfile, join
from uci import Uci, UciExceptionNotFound, UciException
import json
import logging
import re
import threading
from . import internal_funcs

from .common import *

log = logging.getLogger('werkzeug')


def formatUCI(uciDict):
    # print("Dict")
    # print(uciDict)
    resultDict = {}
    for key in uciDict.keys():
        # print("key")
        # print(key)
        # print(type(uciDict[key]))
        if type(uciDict[key]) is dict:
            # print("is dict")
            result = formatUCI(uciDict[key])
            for key2 in result.keys():
                resultDict[key + "." + key2] = result[key2]
        else:
            resultDict[key] = uciDict[key]
    # print("resultDict")
    # print(resultDict)
    return resultDict


def formatUciTupleIntoString(uciTuple):
    uciString = uciTuple[0] + "." + uciTuple[1] + "." + uciTuple[2]
    return uciString


def convertBoolValueToString(value):
    if isinstance(value, bool):
        if value:
            value = "true"
        else:
            value = "false"
    elif isinstance(value, dict):
        for elem in value:
            value[elem] = convertBoolValueToString(value[elem])
    return value


def getAllConfigs():
    mypath = "/barix/config/current"
    config_names = [f for f in listdir(mypath) if isfile(join(mypath, f))]
    return config_names


def getAllUcis():
    ucisList = {}
    uciCli = Uci()
    configsList = getAllConfigs()
    for config in configsList:
        configValue = ''
        try:
            configValue = uciCli.get(config)
        except UciExceptionNotFound:
            log.debug("uci \"{}\" NOT FOUND".format(config))
        except Exception as e:
            log.error(e)
        if configValue != '':
            ucisList[config] = configValue
    return ucisList


# receives a string uci in the format config.section.option
# returns either the uci value, '' if uci is not found and None if an error occurred
def getUci(uci):
    uciCli = Uci()
    path_params = uci.split('.')
    value = None
    try:
        value = uciCli.get(path_params[0], path_params[1], path_params[2])
    except UciExceptionNotFound:
        log.debug("uci \"{}\" NOT FOUND".format(uci))
        value = ''
    except Exception as e:
        log.error(e)
    return value


"""
Get the value from a provided uci
If only config is provided, it will return a dictionary of dictionaries where:
    - first level keys are section keys and the values are dictionaries with section content.
    - second level keys are options and lists from that section. If it is an option, the value is either a string and or a tuple containing strings.
If only config and section is provided, it will return a dictionary with all options and lists in that section.
If config, section and option are provided, then it will return either a string or a tuple of strings if it consists on a list.
If any requested config, section or option are not found then UciExceptionNotFound is thrown and an empty value will be returned.
If any other exception is thrown, the returned value is None.

"""


def getUciValue(config, section=None, option=None):
    error = ALL_OK
    value = None

    uciCli = Uci()

    # get only with config
    if config is not None and section is None and option is None:
        try:
            value = uciCli.get(config)
        except UciExceptionNotFound:
            log.debug("config \"{}\" NOT FOUND".format(config))
        except Exception as e:
            log.error("Error occurred while getting config \"{}\" value: {}".format(config, e))
            error = EXCEPTION_OCCURRED_ERROR_CODE
    elif config is not None and section is not None and option is None:
        try:
            value = uciCli.get(config, section)
        except UciExceptionNotFound:
            log.debug("section \"{}.{}\" NOT FOUND".format(config, section))
        except Exception as e:
            log.error("Error occurred while getting section \"{}.{}\" value: {}".format(config, section, e))
            error = EXCEPTION_OCCURRED_ERROR_CODE
    elif config is not None and section is not None and option is not None:
        try:
            value = uciCli.get(config, section, option)
        except UciExceptionNotFound:
            log.debug("uci \"{}.{}.{}\" NOT FOUND".format(config, section, option))
            value = ''
        except Exception as e:
            log.error("Error occurred while getting uci \"{}.{}.{}\" value: {}".format(config, section, option, e))
            error = EXCEPTION_OCCURRED_ERROR_CODE

    return value, error


"""
Set uci value
If option is not passed, defaults to None and the entire section value will be set. Value must be a dictionary.
If option is passed, value must be a string or a list, and option value will be set, instead of section.
If verify is True, it will check if value changed before setting. If it didn't change, it won't set the new value.
If verify is not passed, it will default to False, and the value will be set, even if it didn't changed.
uciCli is also an argument because if one uses a new uciCli each value set before commiting, previous changes will be lost. If uciCli is none, a new one will be used.
"""


def setUci(config, section, option, value, verify=False, uciCli=None):
    # TODO: set only config, set only config and section
    canSet = True
    uciSet = False

    if uciCli == None:
        uciCli = Uci()

    value = convertBoolValueToString(value)

    if verify:
        try:
            curr_value, error = getUciValue(config, section, option)
            if error == ALL_OK:
                if isinstance(value, dict):
                    if curr_value != '':
                        canSet = False
                        curr_value_json = json.loads(curr_value)
                        for newKey in value.keys():
                            if newKey not in curr_value_json.keys():
                                canSet = True
                                break
                            else:
                                if curr_value_json[newKey] != value[newKey]:
                                    canSet = True
                                    break
                        for oldKey in curr_value_json.keys():
                            if oldKey not in value.keys():
                                canSet = True
                                break
                else:
                    if curr_value == str(value):
                        # log.debug("Value from uci '{}.{}.{}' hasn't changed, no need to set".format(config, section, option))
                        canSet = False
            else:
                return uciCli, uciSet, error
        except Exception as e:
            log.error("Error while setting uci {}.{}.{} with value {} : {}".format(config, section, option, value, e))
            return uciCli, uciSet, EXCEPTION_OCCURRED_ERROR_CODE

    if canSet:
        try:
            if isinstance(value, dict):
                value = json.dumps(value)
            log.debug("set uci {}.{}.{}, value {}".format(config, section, option, value))
            uciCli.set(config, section, option, str(value))
            uciSet = True
        except UciException:  # invalid uci
            log.error("Error while setting uci {}.{}.{} with value {} : invalid".format(config, section, option, value))
            return uciCli, uciSet, INVALID_PARAMETER_ERROR_CODE
        except Exception as e:
            log.error("Error while setting uci {}.{}.{} with value {} : {}".format(config, section, option, value, e))
            return uciCli, uciSet, EXCEPTION_OCCURRED_ERROR_CODE

    return uciCli, uciSet, ALL_OK


def revertUci(uciTuple, uciCli=None):
    error = ALL_OK

    if uciCli == None:
        uciCli = Uci()

    try:
        log.debug("Reverting uci {}".format(uciTuple))
        uciCli.revert(uciTuple[0], uciTuple[1], uciTuple[2])
    except Exception as e:
        log.error("It was not possible to revert changes made to uci {}: {}".format(uciTuple, e))
        error = EXCEPTION_OCCURRED_ERROR_CODE
    return error


"""
#uciList example: {'config.section.option' : value, 'config.section.option' : value, ...}
def setUcisGeneral(uciList, commit, restart):
    return_msg = ''
    return_code = ALL_OK

    uciCli = Uci()

    if uciList != {}:  # if list of ucis to set is not empty
        uciListSet = []
        for uci in uciList:
            log.debug("uci: {}".format(uci))
            # verify if params are valid (if the number of params is correct and exist)
            uciParams = uci.split('.')
            if len(uciParams) != 3:
                return_msg = INVALID_PARAMETER_ERROR_MSG.format(uci)
                return_code = INVALID_PARAMETER_ERROR_CODE
                break
            try:
                curr_value = getUci(uci)
                if not curr_value == None:
                    if curr_value == str(uciList[uci]):
                        log.debug("value hasn't changed, no need to set")
                    else:
                        log.debug("uciCli.set({}.{}.{} {})".format(uciParams[0], uciParams[1], uciParams[2], uciList[uci]))
                        uciCli.set(uciParams[0], uciParams[1], uciParams[2], str(uciList[uci]))
                        uciListSet.append(uci)
                else:
                    return_msg = ''
                    return_code = EXCEPTION_OCCURRED_ERROR_CODE
                    break
            except UciException:
                log.error("It was not possible to set uci {} with value {}".format(uci, str(uciList[uci])))
                return_msg = INVALID_PARAMETER_ERROR_MSG.format(uci)
                return_code = INVALID_PARAMETER_ERROR_CODE
                break
            except Exception as e:
                log.debug(e)
                return '', EXCEPTION_OCCURRED_ERROR_CODE
        if return_code == ALL_OK:
            log.debug('Should commit: {}'.format('commit'))
            if commit and uciListSet != []:
                configsList = []
                for uci in uciListSet:
                    config = uci.split('.')[0]
                    configsList.append(config)
                configsToCommit = set(configsList)
                result = commitUcis(configsToCommit, uciCli)
                if result == EXCEPTION_OCCURRED_ERROR_CODE:
                    return '', EXCEPTION_OCCURRED_ERROR_CODE
            if restart:
                result,error = internal_funcs.createListOfServicesToRestart(uciListSet)
                if error != ALL_OK:
                    return '', EXCEPTION_OCCURRED_ERROR_CODE
                if result != 0:
                    result = launchRestartServicesThread()
                    if result != ALL_OK:
                        return '', EXCEPTION_OCCURRED_ERROR_CODE
            #return return_msg, return_code
            return uciListSet, return_code
    else:
        return_msg = "Uci List to set is empty"
        log.error(return_msg)
        return_code = INVALID_PARAMETER_ERROR_CODE
    return return_msg, return_code
"""

"""
Iterates over a JSON dict of ucis and sets the corresponding value.
uciCli is returned to execute next uci operations under the same scope. If next uciCli would be a new one, previous uci changes (current operation) would be lost.
Returns:
 - error code: ALL_OK, INVALID_PARAMETER_ERROR_CODE or EXCEPTION_OCCURRED_ERROR_CODE)
 - reason: uci (only for INVALID_PARAMETER_ERROR_CODE) or None
 - uciCli
 - configsToCommit
 - ucisConfigured
"""


def setNewUciConfigs(jsonUcis, shouldCommit, shouldRestartServices, additionalActions=None):
    log.debug("Ucis to set: {}".format(jsonUcis))
    ucisConfigured = []  # in case of failure, to revert
    configsToCommit = set()  # to commit ucis
    uciCli = None
    error = ALL_OK
    reason = None
    for uci in jsonUcis:
        params = uci.split('.')
        if len(params) == 3:
            # config = params[0], section = params[1], option = params[2], value = jsonUcis[uci]
            uciCli, uciSet, error = setUci(params[0], params[1], params[2], jsonUcis[uci], True, uciCli)
            if error == ALL_OK:
                if uciSet:
                    ucisConfigured.append((params[0], params[1], params[2]))
                    configsToCommit.add(params[0])
            else:
                # revert everything done so far
                for uciTuple in ucisConfigured:
                    revert_error = revertUci(uciTuple, uciCli)
                    if not (revert_error == ALL_OK):
                        return revert_error, reason, ucisConfigured
                if error == INVALID_PARAMETER_ERROR_CODE:
                    reason = uci
                return error, reason, ucisConfigured
    # commit uci changes?
    if shouldCommit:
        # uci changes must be committed
        error = commitUcis(configsToCommit, uciCli)
        if error != ALL_OK:
            return error, reason, ucisConfigured

    # perform additional actions before restarting services
    if additionalActions is not None:
        additionalActions(ucisConfigured)

    # restart services?
    if shouldRestartServices:

        # add services affected by uci changes to file with restart list stored
        result, error = internal_funcs.createListOfServicesToRestart(ucisConfigured)
        if error != ALL_OK:
            return error, reason, ucisConfigured

        # restart services
        if result != 0:
            result = launchRestartServicesThread()
            if result != ALL_OK:
                return error, reason, ucisConfigured

    return error, reason, ucisConfigured


def setAndCommitUcis(jsonUcis, shouldCommit):
    log.debug("Ucis to set: {}".format(jsonUcis))
    ucisConfigured = []  # in case of failure, to revert
    configsToCommit = set()  # to commit ucis
    uciCli = None
    error = ALL_OK
    reason = None
    for uci in jsonUcis:
        params = uci.split('.')
        if len(params) == 3:
            # config = params[0], section = params[1], option = params[2], value = jsonUcis[uci]
            uciCli, uciSet, error = setUci(params[0], params[1], params[2], jsonUcis[uci], True, uciCli)
            if error == ALL_OK:
                if uciSet:
                    ucisConfigured.append((params[0], params[1], params[2]))
                    configsToCommit.add(params[0])
            else:
                # revert everything done so far
                for uciTuple in ucisConfigured:
                    error = revertUci(uciTuple, uciCli)
                    if not error == ALL_OK:
                        return error, reason
                if error == INVALID_PARAMETER_ERROR_CODE:
                    reason = uci
                return error, reason, ucisConfigured

    # commit uci changes?
    if shouldCommit:
        # uci changes must be committed
        error = commitUcis(configsToCommit, uciCli)
        if error != ALL_OK:
            return error, reason, ucisConfigured

    return error, reason, ucisConfigured


def commitUcis(configsList, uciCli):
    for config in configsList:
        try:
            log.debug("Commiting config {}".format(config))
            uciCli.commit(config)
        except Exception as e:
            # should never happen!!
            # return_msg = {"error_code": COMMIT_UCI_ERROR_CODE, "msg": "It was not possible to commit config {}".format(config)}
            log.error("Committing config {} failed: {}".format(config, e))
            # error code: Internal Server Error
            return EXCEPTION_OCCURRED_ERROR_CODE
    return ALL_OK


def launchRestartServicesThread():
    log.debug('Launching thread to restart services')
    try:
        # t = threading.Thread(target=internal_funcs.addConfigToRestartList, args=(uciList,))
        t = threading.Thread(target=internal_funcs.restartServices)
        t.start()
    except Exception as e:
        log.error(e)
        return EXCEPTION_OCCURRED_ERROR_CODE
    return ALL_OK


""" pyuci """

"""
## when config does not exist
>>> uci.get('testconfig')
Traceback (most recent call last):
  File "<stdin>", line 1, in <module>
uci.UciExceptionNotFound

## when config exist but is empty: 
>>> uci.get('testconfig')
{}

>>> uci.get('testconfig', 'section1')
Traceback (most recent call last):
  File "<stdin>", line 1, in <module>
uci.UciExceptionNotFound

## when config and section exist, but section is empty:
>>> uci.get('testconfig')
{'section1': {}}

>>> uci.get('testconfig', 'section1')
'section'


"""


def validateUciValues(jsonUcis):
    for uci in jsonUcis:
        if not validateUciValue(uci, jsonUcis[uci]):
            return False
    return True


def validateUciValue(uci, value):
    HPN_URL_REGEX = "^rtp:\/\/((2(?:2[4-9]|3\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]\d?|0)){3}:)|@:|0.0.0.0:)(102[4-9]|10[3-9][0-9]|1[1-9][0-9]{2}|[2-9][0-9]{3}|[1-5][0-9]{4}|6[0-4][0-9]{3}|65[0-4][0-9]{2}|655[0-2][0-9]|6553[0-5])(\?format=\w+&channels=\d&rate=\d+(&pt=\d+)?)?$"
    BGM_URL_REGEX = "(^$)|(^rtp:\/\/((2(?:2[4-9]|3\d)(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]\d?|0)){3}:)|@:|0.0.0.0:)(102[4-9]|10[3-9][0-9]|1[1-9][0-9]{2}|[2-9][0-9]{3}|[1-5][0-9]{4}|6[0-4][0-9]{3}|65[0-4][0-9]{2}|655[0-2][0-9]|6553[0-5])(\?format=\w+&channels=\d&rate=\d+(&pt=\d+)?)?$)"
    INT_NUMBER_REGEX = "^-?\d+$"
    HOST_REGEX = "(^(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5]).){3}([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])$)|(^(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9-]*[a-zA-Z0-9]).)*([A-Za-z0-9]|[A-Za-z0-9][A-Za-z0-9-]*[A-Za-z0-9])$)"
    SIP_DOMAIN_REGEX = "^([0-9]|[1-8][0-9]|9[0-9]|1[0-9]{2}|2[01][0-9]|22[0-3])(?:\.(?:25[0-5]|2[0-4]\d|1\d\d|[1-9]\d?|0)){3}$|^(([a-zA-Z0-9-]+\.)*[a-zA-Z]+)$"
    DECIMAL_NUMBER_REGEX = "^-?\d+(\.\d+)?$"
    ALIAS_REGEX = "^[\w\s#|&\/.,\-]*$"
    DESCRIPTION_REGEX = "^.*$"
    GENERAL_IP_ADDRESS_REGEX = "(^$)|(^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$)"
    NUMBER_INTERVAL_REGEX = "^([1-6](,[1-6])*)|()$"
    SERIAL_CONFIG_REGEX = "^([3-9][0-9]{2}|[1-9][0-9]{3,4}|10[0-9]{4}|11[0-4][0-9]{3}|115[0-1][0-9]{2}|115200):(E|O|N):[7-8]:[1-2]$"
    HTTP_URL_REGEX = "^http[s]?:\/\/(.+)$"
    SIP_AUTODIAL_REGEX = "(^$)|(^(sip:\/\/.+@([a-zA-Z]+\.)*[a-zA-Z]+)|([a-zA-Z0-9])+$)"
    IP_ADDR_AND_PORT = "(^$)|(^(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5]).){3}([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5]):[0-9]+$)"
    NTP_URL_REGEX = "(^$)|(^(http[s]*:\/\/)?(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9-]*[a-zA-Z0-9]).)*([A-Za-z0-9]|[A-Za-z0-9][A-Za-z0-9-]*[A-Za-z0-9])$)"
    if uci == "epic.bgm.url":
        if re.fullmatch(BGM_URL_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.hpn.url":
        if re.fullmatch(HPN_URL_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.sip.call_timeout" or uci == "epic.sip.stream_timeout" or uci == "epic.sip.answer_time":
        if re.fullmatch(INT_NUMBER_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.sip.micin_gain":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and 0 <= int(value) <= 42):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.hpn.volume" or uci == "epic.bgm.volume" or uci == "epic.sip.volume" or uci == "epic.sip.input_gain" \
            or uci == "epic.main.volume":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and 0 <= int(value) <= 100):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.serialgw.host":
        if re.fullmatch(HOST_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ntp.source.server1" or uci == "ntp.source.server2" or uci == "ntp.source.server3":
        if re.fullmatch(NTP_URL_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.serialgw.port" or uci == "epic.api.abcl_tcp_port":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and 1024 <= int(value) <= 65535):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.sip.domain":
        if re.fullmatch(SIP_DOMAIN_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.sip.hdx_level":
        if not (re.fullmatch(DECIMAL_NUMBER_REGEX, str(value)) is not None and 0.0 <= float(value) <= 1.0):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.sip.hdx_timeout":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and 10 <= int(value) <= 2000):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.main.device_alias":
        if re.fullmatch(ALIAS_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.main.description":
        if re.fullmatch(DESCRIPTION_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ae_ms.volume.out0" or uci == "ae_ms.volume.out1" or uci == "ae_ms.volume.out2" \
            or uci == "ae_ms.volume.out3" or uci == "ae_ms.volume.out4" or uci == "ae_ms.mixer.ch1" \
            or uci == "ae_ms.mixer.ch2" or uci == "ae_ms.mixer.ch3" or uci == "ae_ms.mixer.ch4" \
            or uci == "ae_ms.mixer.ch5" or uci == "ae_ms.mixer.ch6" or uci == "ae_ms.mixer.ch100"\
            or uci == "ae_ms.talkback.level_wpa" or uci == "ae_ms.talkback.level_rp1":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and -79 <= int(value) <= 16):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "network.eth0.ipaddr" or uci == "network.eth0.netmask" or uci == "network.eth0.gateway" \
            or uci == "network.eth0.dns1" or uci == "network.eth0.dns2":
        if re.fullmatch(GENERAL_IP_ADDRESS_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ae_ms.mixer.enable1" or uci == "ae_ms.mixer.enable2" or uci == "ae_ms.mixer.enable3" \
            or uci == "ae_ms.mixer.enable4" or uci == "ae_ms.mixer.enable5" or uci == "ae_ms.mixer.enable6" \
            or uci == "ae_ms.mixer.enable100" or uci == "ae_ms.volume.enable0" or uci == "ae_ms.volume.enable1" \
            or uci == "ae_ms.volume.enable2" or uci == "ae_ms.volume.enable3" or uci == "ae_ms.volume.enable4"\
            or uci == "ae_ms.eq.enable" or uci == "epic.main.fw_update_auto" or uci == "epic.hpn.high_prio"\
            or uci == "epic.sip.enable" or uci == "epic.sip.auto_answer" or uci == "epic.sip.beep_on_answer"\
            or uci == "epic.sip.half_duplex" or uci == "epic.sip.aec" or uci == "epic.io.notify_relay1" \
            or uci == "epic.io.notify_relay2" or uci == "epic.io.notify_relay3" or uci == "epic.io.notify_relay4" \
            or uci == "ae_ms.apcore.enable" or uci == "ae_ms.serial.csp_1" or uci == "ae_ms.serial.csp_2" \
            or uci == "ae_ms.serial.csp_3" or uci == "ae_ms.serial.csp_4" or uci == "ae_ms.serial.csp_5" \
            or uci == "ae_ms.poe.override" or uci == "ae_ms.banzai.enable" or uci == "epic.snmp.enable":
        if not (isinstance(value, bool) or value == 'true' or value == 'false'):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ae_ms.eq.band1" or uci == "ae_ms.eq.band2" or uci == "ae_ms.eq.band3" or uci == "ae_ms.eq.band4" \
            or uci == "ae_ms.eq.band5":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and -10 <= int(value) <= 10):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ae_ms.serial.egress_list_1" or uci == "ae_ms.serial.egress_list_2" \
            or uci == "ae_ms.serial.egress_list_3" or uci == "ae_ms.serial.egress_list_4" \
            or uci == "ae_ms.serial.egress_list_5" or uci == "ae_ms.serial.egress_list_6":
        if re.fullmatch(NUMBER_INTERVAL_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ae_ms.serial.config_1" or uci == "ae_ms.serial.config_2" or uci == "ae_ms.serial.config_3" \
            or uci == "ae_ms.serial.config_4" or uci == "ae_ms.serial.config_5":
        if re.fullmatch(SERIAL_CONFIG_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ae_ms.arcanix.mode":
        if not (str(value).lower() == "enabled" or str(value).lower() == "disabled"):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.main.fw_update_url":
        if re.fullmatch(HTTP_URL_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.bgm.buffer_size" or uci == "epic.hpn.buffer_size":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and 50 <= int(value) <= 10000):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.sip.transport":
        if not (str(value).lower() == "udp" or str(value).lower() == "tcp"):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.sip.answer_time" or uci == "epic.sip.call_timeout" or uci == "epic.sip.stream_timeout":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and 0 <= int(value) <= 600):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.sip.autodial1" or uci == "epic.sip.autodial2" or uci == "epic.sip.autodial3" or uci == "epic.sip.autodial4":
        if re.fullmatch(SIP_AUTODIAL_REGEX, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.serialgw.mode":
        if not (str(value).lower() == "disabled" or str(value).lower() == "server" or str(value).lower() == "client"):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.serialgw.baud":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and (int(value) == 9600 or int(value) == 19200 or int(value) == 38400 or int(value) == 57600 or int(value) == 115200)):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.serialgw.parity":
        if not (str(value).lower() == "none" or str(value).lower() == "even" or str(value).lower() == "odd"):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.serialgw.databits":
        if not (str(value) == "7" or str(value) == "8"):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.serialgw.stopbits":
        if not (str(value) == "1" or str(value) == "2"):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ae_ms.apcore.fixed_client":
        if re.fullmatch(IP_ADDR_AND_PORT, str(value)) is None:
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ae_ms.banzai.threshold":
        if not (re.fullmatch(INT_NUMBER_REGEX, str(value)) is not None and -80 <= int(value) <= -1):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "ae_ms.speaker.impedance":
        if not (str(value) == "2" or str(value) == "4" or str(value) == "8"):
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    elif uci == "epic.sip.input":
        deviceModelFile = "/barix/apps/epic-sip/MODEL"
        deviceModel = "MS-500A"
        if path.isfile(deviceModelFile):
            try:
                f = open(deviceModelFile, 'r')
                content = f.read().strip('\r\n')
                if content == "MS300B":
                    deviceModel = "MS-300B"
                elif content == "MS500A":
                    deviceModel = "MS-500A"
                elif content == "MS700":
                    deviceModel = "MS-700"
                else:
                    deviceModel = content
                f.close()
            except Exception as e:
                log.error("Error reading file {}: {}".format(deviceModelFile, e))
                raise e
        if deviceModel == "MS700":
            if str(value).upper() != "LINEIN":
                log.error("Value {} for {} is invalid".format(value, uci))
                return False
        else:
            if not (str(value).upper() == "LINEIN" or str(value).upper() == "MICIN"):
                log.error("Value {} for {} is invalid".format(value, uci))
                return False
    elif uci == "dropbear.RunCtl.enable":
        if value != "1" and value != "0" and value != "on_until_reboot":
            log.error("Value {} for {} is invalid".format(value, uci))
            return False
    return True
