#!/bin/sh

#!/bin/sh 

# some basic functions to control the TAS-2770/TAS-6421 amplifiers
# from user space when the tas2770 is not compiled in, or not working
# Requires i2c-tools to be installed
# 
# The amplifier needs to be initialized before doing anything else
# Sequence:
# 1. Source this file. It will make the functions accessible, and prepare
#    the registers from page 0, book 0 to be accessible for writing
# 2. reset the amplifier (call reset_amp)
# 3. call the init_amp function with the start volume as parameter.
#    If not specified, it sets the minimum volume (11 dBV)
# 4. Adjust volume separately (set_pcm_vol and set_tas2771_vol) and finally start the amp (power_on_amp)
# 5. 3 and 4 can be done in one step calling init_start_tas2770
#
# Notes:
# 1. Once the amplifier is initialized, you can change the volume on the fly
# 2. When audio is stopped, and there is nothing to keep the i2s interface
#    "busy" (ex like playing silent loop), so tas-2771 amp has to be initialized 
#    again the next time the audio starts. TAS-6421 is not affected by that


CHIP_ADDR="0x41"
SET2C="/usr/sbin/i2cset -y 0 $CHIP_ADDR"
GET2C="/usr/sbin/i2cget -y 0 $CHIP_ADDR"

# set TAS2770 register
set_reg() {        
    #echo "Executing $SET2C $1 $2"
    eval "$SET2C $1 $2"
}

get_reg() {                                                                                     
    ret=$(eval "$GET2C $1")
	echo "$ret"                                                                            
} 

set_tas2771_vol() {
	vol=0x00 #${I2C_TAS2771_VOL["$1"]}
	#echo "Setting TAS2771 Amplifier Volume: $1 to $vol"
	set_reg 0x03 "$vol"
}

# pass the PCM volume from 0dB to -100dB in steps of 0.5 dB
# anything below -100 dB mutes it
# ex. call: set_pcm_vol -21.5
set_tas2770_pcm_vol() {
	pcm_vol=$(awk -v val=$1 'function abs(v) {return v < 0 ? -v : v} BEGIN{printf("0x%02x", abs(val)*2)}')
	#echo "Setting PCM Volume: $1 to $pcm_vol"
	set_reg 0x05 "$pcm_vol"
}

set_pcm_vol() {
	#echo "Requested volume: $1"
    set_tas2770_pcm_vol "$1"
}

get_pcm_vol() {                                                                                 
    raw_hex_vol=$(get_reg "0x05")
	raw_vol=$(printf "%d" "$raw_hex_vol")
	ret_pcm_vol="0 dB"	

	ret_pcm_vol=$(awk -v val=$raw_vol 'function calc(v) {return v = 0 ? 0 : -v} BEGIN{printf("%.1f dB", calc(val)/2)}')                                                                                             
	echo -n "$ret_pcm_vol"
}

power_on_amp() {
	# powering up the amp
    set_reg 0x02 0x0c
    #echo "TAS2770 amplifier powered on!"
}

init_start_tas2770() {
	init_tas2770 "$1"
	power_on_amp
}

init_amp() {
    init_tas2770 "$1"
}

init_tas2770() {
	# setting registers
	#set_reg 0x03 0x14
	set_reg 0x04 0x00
	#mute the PCM Volume at start
	set_reg 0x05 0xff
	
	set_reg 0x07 0x02
	set_reg 0x08 0x40
	set_reg 0x0a 0x37
	set_reg 0x0b 0x03
	set_reg 0x0c 0x10
	set_reg 0x0e 0xf3
	set_reg 0x15 0x15
	set_reg 0x18 0x4b
	set_reg 0x1a 0x10
	set_reg 0x32 0x80
	set_reg 0x3c 0x18
	set_reg 0x7e 0x19
	#setting amplifier volume to intermediate value (18.0 dBV)
	set_tas2771_vol "110"
	# set pcm_volume if requested
	if [ "$1" != "" ]; then
		set_pcm_vol "$1"
	fi
	
	sleep 0.002
	#echo "TAS2770 I2C init DONE !"
}

# prepare the i2c regs to write
# We mostly write to page 0, book 0, so ignore 
# page 1 for now
prepare_tas2770_regs() {
	# set page 0
	set_reg 0x00 0x00
	# set book 0
	set_reg 0x7f 0x00
	# set again page 0
	set_reg 0x00 0x00
}

prepare_regs() {
		prepare_tas2770_regs
}

# reset the TAS2770 device
# always call this function after setting 
# the page to write and before the init
reset_amp() {
    set_reg 0x01 0x01
}

###################################################################

prepare_regs
reset_amp
# apply some safe defaults for volume
vol="-35"
sleep 0.002
init_amp "$vol"
set_pcm_vol "$vol"
sleep 0.002
# powering up the amp
power_on_amp