#!/bin/sh

# Barix GPIO utilities.
# Copyright (C) 2018 Barix AG (http://www.barix.com). All rights reserved.
#
# Dependencies from 'qiba-exstreamerl.sh' main script:
#  Includes
#   + 'qiba-log.sh': Logging facilities.
#  Variables
#   + 'TEST_HOST: Testing on host.

# The sys path root.
# Allowed to be specified externally, e.g., for testing.
[ -z "${SYSPATH}" ] && SYSPATH=""

# The log messages prefix.
# Allowed to be specified externally.
[ -z "${LOG_PFX}" ] && LOG_PFX="qiba-gpio"

# The sys path for GPIO devices.
readonly SYSPATH_GPIO="${SYSPATH}/sys/class/gpio"

# GPIO: Use the GPIO sysfs to export/unexport GPIO numbers.
#
# Parameters:
#  $1: The GPIO base number.
#  $2: The number of GPIOs.
#  $3: What to do with the GPIOs: 'export' or 'unexport'.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
_gpio_exp_action() {
	local base="${1}"
	local num="${2}"
	local action="${3}"

	local i gpio res

	local ret=0
	local retf=0
	local f="${FUNCNAME[0]}"

	for i in $(seq 0 $((${num} - 1))); do
		gpio=$((${base} + ${i}))
		ret=0
		if [ "${TEST_HOST}" = "1" ]; then
			res=$(echo "${gpio}" 2>&1 >> "${SYSPATH_GPIO}/${action}") || ret=$?
		else
			res=$(echo "${gpio}" 2>&1 > "${SYSPATH_GPIO}/${action}") || ret=$?
		fi
		if [ ${ret} -ne 0 ]; then
			log_err "${f}" "Can't ${action} GPIO '${gpio}' (${ret}):"
			log_err "${f}" "${res}"
			if [ ${retf} -eq 0 ]; then
				retf=${ret}
			fi
		fi
	done

	return ${retf}
}

# GPIO: Write value to GPIOs sysfs file.
#
# Parameters:
#  $1: The GPIO base number where to write.
#  $2: The number of GPIOs to write.
#  $3: The value to write.
#  $4: The GPIO sysfs file to write.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
_gpio_sys_write() {
	local base="${1}"
	local num="${2}"
	local value="${3}"
	local file="${4}"

	local i gpio gpio_p res

	local ret=0
	local retf=0
	local f="${FUNCNAME[0]}"

	for i in $(seq 0 $((${num} - 1))); do
		ret=0
		gpio=$((${base} + ${i}))
		gpio_p="${SYSPATH_GPIO}/gpio${gpio}/${file}"
		res=$(echo "${value}" 2>&1 > "${gpio_p}") || ret=$?
		if [ ${ret} -ne 0 ]; then
			log_err "${f}" \
				"Can't write '${file}' on GPIO '${gpio}' (${ret}):"
			log_err "${f}" "${res}"
			if [ ${retf} -eq 0 ]; then
				retf=${ret}
			fi
		fi
	done

	return ${retf}
}

# GPIO: Get GPIO data (base number and number of GPIOs).
#
# Parameters:
#  $1: The sys path where to search for the GPIO controller info.
#   E.g., in case of a I2C GPIO Expander attached to I2C bus 0 at address
#   0x20, this path would be '/sys/bus/i2c/devices/0-0020'.
#  $2: [opt] Variable to store the GPIO base number output.
#  $3: [opt] Variable to store the number of GPIOs output.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <outp1>: The GPIO base number, placed in var specified by param $1.
#  <outp2>: The Number of GPIOs, placed in var specified by param $2.
#  <echo>: The GPIO data, in the form of '<base> <ngpio>', where <base> is the
#   base GPIO number and <ngpio> is the number of GPIOs, e.g. '400 16'. Echo
#   output only sent if none of params $2 and $3 specified.
gpio_data_get() {
	local sys_p="${1}"
	local outp1="${2}"
	local outp2="${3}"

	local chip base_f ngpio_f nbase ngpio

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ -z "${sys_p}" ]; then
		log_err "${f}" "Missing GPIO sys path parameter"
		return 1
	fi

	# If the sys path is for a GPIO controller, it must have a 'gpio' dir.
	sys_p="${sys_p}/gpio"

	# If the external GPIO driver is properly setup, there will be a single dir
	# entry in ${sys_p}, which is the GPIO chip dir.
	chip=$(ls "${sys_p}" 2>&1) || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "No GPIO chip found in '${sys_p}' (${ret}):"
		log_err "${f}" "${chip}"
		return ${ret}
	fi

	# The GPIO chip sys dir has files for the base GPIO and the number of GPIOs.
	sys_p="${sys_p}/${chip}"

	base_f="${sys_p}/base"
	nbase=$(cat "${base_f}" 2>&1) || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get GPIO base (${ret}):"
		log_err "${f}" "${nbase}"
		return ${ret}
	fi
	if [ -z "${nbase}" ]; then
		log_err "${f}" "Got empty GPIO base at:"
		log_err "${f}" "${base_f}"
		return 1
	fi

	ngpio_f="${sys_p}/ngpio"
	ngpio=$(cat "${ngpio_f}" 2>&1) || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get GPIO ngpio (${ret}):"
		log_err "${f}" "${ngpio}"
		return ${ret}
	fi
	if [ -z "${ngpio}" ]; then
		log_err "${f}" "Got empty GPIO ngpio at:"
		log_err "${f}" "${ngpio_f}"
		return 1
	fi

	# If there is a variable to store the output, use it. Otherwise just
	# return the output through echoing.
	if [ -n "${outp1}" ]; then
		eval ${outp1}="'${nbase}'"
		[ -n "${outp2}" ] && eval ${outp2}="'${ngpio}'"
		return 0
	fi
	echo "${nbase} ${ngpio}"
	return 0
}

# GPIO: Use the GPIO sysfs to export GPIO numbers.
#
# Parameters:
#  $1: The GPIO base number to export.
#  $2: [opt] The number of GPIOs to export. If not specified, 1 is assumed.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpio_export() {
	local base="${1}"
	local num="${2}"

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ -z "${base}" ]; then
		log_err "${f}" "Missing GPIO base number"
		return 1
	fi
	if [ -z "${num}" ]; then
		num="1"
	fi

	_gpio_exp_action "${base}" "${num}" "export" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Could not export all '${num}' GPIOs (${ret})"
		return ${ret}
	fi

	return 0
}

# GPIO: Use the GPIO sysfs to unexport GPIO numbers.
#
# Parameters:
#  $1: The GPIO base number to unexport.
#  $2: [opt] The number of GPIOs to unexport. If not specified, 1 is assumed.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpio_unexport() {
	local base="${1}"
	local num="${2}"

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ -z "${base}" ]; then
		log_err "${f}" "Missing GPIO base number"
		return 1
	fi
	if [ -z "${num}" ]; then
		num="1"
	fi

	_gpio_exp_action "${base}" "${num}" "unexport" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Could not unexport all '${num}' GPIOs (${ret})"
		return ${ret}
	fi

	return 0
}

# GPIO: Use the GPIO sysfs to set GPIOs direction.
#
# Parameters:
#  $1: The GPIO direction, 'in' or 'out'.
#  $2: The GPIO base number.
#  $3: [opt] The number of GPIOs. If not specified, 1 is assumed.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpio_dir_set() {
	local dir="${1}"
	local base="${2}"
	local num="${3}"

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ "${dir}" != "in" ] && [ "${dir}" != "out" ]; then
		log_err "${f}" "Invalid GPIO direction '${dir}'"
		return 1
	fi
	if [ -z "${base}" ]; then
		log_err "${f}" "Missing GPIO base number"
		return 1
	fi
	if [ -z "${num}" ]; then
		num="1"
	fi

	_gpio_sys_write "${base}" "${num}" "${dir}" "direction" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Could not set direction in all '${num}' GPIOs (${ret})"
		return ${ret}
	fi

	return 0
}

# GPIO: Use the GPIO sysfs to get a GPIO direction.
#
# Parameters:
#  $1: The GPIO number.
#  $2: [opt] Variable to store the output.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <echo>|<outp>: The GPIO direction, 'in' or 'out'.
gpio_dir_get() {
	local num="${1}"
	local outp="${2}"

	local _dir

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ -z "${num}" ]; then
		log_err "${f}" "Missing GPIO number"
		return 1
	fi

	gpio_p="${SYSPATH_GPIO}/gpio${num}/direction"
	_dir=$(cat "${gpio_p}" 2>&1) || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get GPIO '${num}' direction (${ret}):"
		log_err "${f}" "${_dir}"
		return ${ret}
	fi

	# If there is a variable to store the output, use it. Otherwise just
	# return the output through echoing.
	[ -n "${outp}" ] && eval ${outp}="'${_dir}'" || echo "${_dir}"

	return 0
}

# GPIO: Use the GPIO sysfs to set GPIOs value.
#
# Parameters:
#  $1: The GPIO value.
#  $2: The GPIO base number.
#  $3: [opt] The number of GPIOs. If not specified, 1 is assumed.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
gpio_val_set() {
	local val="${1}"
	local base="${2}"
	local num="${3}"

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ -z "${base}" ]; then
		log_err "${f}" "Missing GPIO base number"
		return 1
	fi
	if [ -z "${num}" ]; then
		num="1"
	fi

	_gpio_sys_write "${base}" "${num}" "${val}" "value" || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't set value in all GPIOs (${ret})"
		log_err "${f}" "Could not set value in all '${num}' GPIOs (${ret})"
		return ${ret}
	fi

	return 0
}

# GPIO: Use the GPIO sysfs to get a GPIO value.
#
# Parameters:
#  $1: The GPIO number.
#  $2: [opt] Variable to store the output.
#
# Returns:
#  <ret>: 0 on SUCCESS, otherwise error code.
#  <echo>|<outp>: The GPIO value.
gpio_val_get() {
	local num="${1}"
	local outp="${2}"

	local _val

	local ret=0
	local f="${FUNCNAME[0]}"

	if [ -z "${num}" ]; then
		log_err "${f}" "Missing GPIO number"
		return 1
	fi

	gpio_p="${SYSPATH_GPIO}/gpio${num}/value"
	_val=$(cat "${gpio_p}" 2>&1) || ret=$?
	if [ ${ret} -ne 0 ]; then
		log_err "${f}" "Can't get GPIO '${num}' value (${ret}):"
		log_err "${f}" "${_val}"
		return ${ret}
	fi

	# If there is a variable to store the output, use it. Otherwise just
	# return the output through echoing.
	[ -n "${outp}" ] && eval ${outp}="'${_val}'" || echo "${_val}"

	return 0
}
