import logging
import time
from threading import Thread
from urllib.parse import urlparse
import baco

logger = logging.getLogger(__name__)

class BacoPlayer:
    """Create a player using Baco library individual modules"""
    def __init__(self, url, deviceName, sampleRate):
        self.url                = url
        self.deviceName         = deviceName
        self.readTimeout        = 500
        self.inputFifoDuration  = 0
        self.outputFifoDuration = 0
        self.outputFormat       = baco.FrameFormat(baco.ChannelLayout.STEREO, baco.SampleFormat.I16 ,sampleRate)

        self.inputFormat        = ""
        self.inputFormatOptions = {}

        self.playerThread       = None
        self.callback           = None
        self.streamInput        = None
        self.decoderChain       = None
        self.equalizerConfig    = None
        self.equalizer          = None
        self.equalizerEnabled   = False
        self.alsaSink           = None
        self.stopPlayer         = False
        self.alive              = False

    def setReadTimeout(self, timeout):
        self.readTimeout = timeout

    def setInputFifoDuration(self, inputFifoDuration):
        self.inputFifoDuration = inputFifoDuration

    def setOutputFifoDuration(self, outputFifoDuration):
        self.outputFifoDuration = outputFifoDuration

    def setInputFormat(self, format, sampleRate, channels):
        """ Define stream input options.
        """
        self.inputFormat = format
        self.inputFormatOptions = {"sample_rate": sampleRate,"channels": channels}

    def setCallback(self, func):
        self.callback = func

    def start(self):
        if self.playerThread is None:
            self.playerThread = Thread(target=self.__player)
            logger.info("start player thread")
            self.stopPlayer = False
            self.playerThread.start()

    def __openStream(self):
        par = urlparse(self.url)
        sdpConf = ""
        # RTP stream and has query parameters (ex: format=mulaw&channels=2&rate=24000)
        if par.scheme == 'rtp' and par.query:
            sdpConf = baco.genSdp(self.url)

        self.streamInput = baco.StreamInput()
        self.streamInput.setTimeout(self.readTimeout)

        if sdpConf:
            res = self.streamInput.openSdp(sdpConf)
        else:
            res = self.streamInput.open(self.url, self.inputFormat, self.inputFormatOptions)
        return res

    def __setDecoderChain(self):
        srcInfo = self.streamInput.getSourceInfo()
        streamDesc = srcInfo.getStream(0)
        if srcInfo.getNumStreams() == 0 or not streamDesc:
            logger.warning("Input format is not known!")
            return False
        else:
            if streamDesc.codec() == baco.CodecId.UNKNOWN:
                logger.warning("Unknown CodecId!")
                return False
            
            config = baco.Config(srcInfo.getStream(0), self.outputFormat)
            config.inputFifoDuration = self.inputFifoDuration
            config.outputFifoDuration = self.outputFifoDuration
            self.decoderChain = baco.DecoderChain(config)
            self.decoderChain.mute(True)
            self.decoderChain.setGain(1, 0, 0)
            self.decoderChain.setGainMask(1)
            self.alsaSink = baco.AlsaSink(self.deviceName,self.outputFormat,0,0)
            if self.equalizerConfig is not None:
                self.equalizer = baco.Equalizer(self.outputFormat,self.equalizerConfig)            
                self.decoderChain.attachSink(self.equalizer)
                self.equalizer.setFrameSink(self.alsaSink)
                if self.equalizerEnabled:
                    self.equalizer.enable(True)
            else:
                self.decoderChain.attachSink(self.alsaSink)
            self.decoderChain.startOutput()
        return True

    def __player(self):
        while not self.stopPlayer:
            res = self.__openStream()
            self.stopReading = False
            if res == baco.ReturnCode.OK:
                if self.__setDecoderChain():
                    while not self.stopReading:
                        res = self.streamInput.readPacket(self.decoderChain)
                        if res != baco.ReturnCode.OK:
                            logger.info("Read Packet Error")
                            self.stopReading = True
                        else:
                            if not self.alive:
                                self.alive = True
                                if self.callback is not None:
                                    self.callback(self.alive)
                    logger.info("stop player")
             
            time.sleep(0.2)

            if self.alive:
                self.alive = False
                if self.callback is not None:
                    self.callback(self.alive)

        logger.info("stop player thread")
        self.playerThread = None

    def connectAudio(self):
        if self.decoderChain is not None:
            self.decoderChain.mute(False)

    def disconnectAudio(self):
        if self.decoderChain is not None:
            self.decoderChain.mute(True)

    def isConnected(self):
        return self.alive

    def setEqualizer(self, config:dict):
        """
        Set Equalizer
        :param config: equalizer json config band list
        """
        try:
            self.equalizerConfig = config
            logger.info("Add Equalizer configuration")
        except Exception as e:
            logger.error(f"setEqualizer: {e}")
    
    def setEqualizerFrequency(self,name, f):
        """
        Set Equalizer band filter central frequency
        :name name: band filter name 
        :param f: central frequency
        """
        try:
            self.equalizer.setFrequency(name,f)
            logger.info(f"Set Filter {name} central frequency: {f}")
        except Exception as e:
            logger.error(f"{e}")    
            
    def setEqualizerQFactor(self,name, q):
        """
        Set Equalizer band filter Q-factor
        :name name: band filter name 
        :param q: q-factor
        """
        try:
            self.equalizer.setQFactor(name,q)
            logger.info(f"Set Filter {name} Q-factor: {q}")
        except Exception as e:
            logger.error(f"{e}")


    def setEqualizerGain(self,name, g):
        """
        Set Equalizer band filter gain
        :name name: band filter name 
        :param g: gain
        """
        try:
            self.equalizer.setGain(name,g)
            logger.info(f"Set Filter {name} gain: {g}")
        except Exception as e:
            logger.error(f"{e}")

    def enableEqualizer(self, enable):
        try:
            if self.equalizer is not None:
                self.equalizer.enable(enable)
                self.equalizerEnabled = enable
                logger.info(f"Enable Equalizer: {enable}")
        except Exception as e:
            self.logger.error(e)

    def disconnect(self):
        if self.decoderChain is not None:
            self.decoderChain.mute(True)
        self.stopReading = True
        self.stopPlayer = True
        logger.info("BacoPlayerModule disconnected...")
