import logging
import os
import re
import subprocess
import uuid
import zipfile

from .exceptions import InvalidUploadedFileError, InvalidAudioFileError
from .stored_files import checkIfFileExtensionIsAccepted
from .system_info import getStorageSize
from .utils import readFromJSONFile, writeToJSONFile

log = logging.getLogger('flask-backend')


def getAudioFileDetails(file_list):
    """
        Get some details from audio files: filename, duration and size.
    :param file_list: list of dictionaries, where each dictionary contains the directory
            where the file is and the file itself.
    :return: a list of dictionaries, where each dictionary contains the details requested.
    """

    info_file_list = []
    for elem in file_list:
        file_info = {
            "filename": elem['file'].name,
            "duration": getAudioFileDuration(os.path.join(elem['dir'], elem['file'].name)),
            "size": elem['file'].stat().st_size
        }
        info_file_list.append(file_info)
    return info_file_list


def uploadAudioFile(file_uploaded, destination_path, file_formats_allowed,
                    free_space, audio_files_json_filepath, **kwargs):
    """
        Upload an Audio File and store it in the device.
    :param file_uploaded: audio file uploaded
    :param destination_path: string containing the full path of where the file should be stored
    :param file_formats_allowed: list of strings containing the audio files extensions allowed
    :param free_space: int representing the free space left on the device
    :param audio_files_json_filepath: string containing the full path of the file that will store the information
            about the audio file uploaded.
    :param kwargs: Either a dictionary or a multiple key=value pairs that need to be added
            MUST be different from the rest of the params
    :return: number of audio files stored.

        Validates if the device has enough space to store the audio file. If not, raises an exception.
        Validates if the file uploaded has a valid extension. It can be a zip file as well.
        If the file is invalid, or any file inside the zip file, return an InvalidUploadedFileError exception.
        If everything goes well, the file is stored in the JSON audio file as well.
        If an exception occurs after the file has been stored, remove the file.
    """

    destination_file_path = ""
    uploaded_file_is_zip = False
    try:
        # generate ID to be the name of the file when stored
        file_id = uuid.uuid4().hex
        destination_file_path = os.path.join(destination_path, file_id)
        # save file in right directory
        file_uploaded.save(destination_file_path)
        # check if file size is larger than available free space in storage (because of 5% root dedicated space,
        # file is stored even if available space shows as 0, since backed is being run as root)
        if os.stat(destination_file_path).st_size > free_space:
            raise Exception('No space left on device')
        # check if file is zip
        uploaded_file_is_zip = zipfile.is_zipfile(destination_file_path)
        if uploaded_file_is_zip:
            n_files_stored = __unzipAudioFiles(destination_file_path, destination_path, file_formats_allowed, free_space)
        else:
            checkIfFileExtensionIsAccepted(file_uploaded.filename, file_formats_allowed)
            __storeAudioFile(file_id, file_uploaded.filename, destination_file_path, audio_files_json_filepath, **kwargs)
            n_files_stored = 1
    except Exception as e:
        log.error(e)
        # if file was stored and then some error occurred, file should be removed
        if destination_file_path != "" and os.path.exists(destination_file_path):
            os.remove(destination_file_path)
        if uploaded_file_is_zip:
            raise InvalidUploadedFileError("Invalid zip file's content")
        else:
            raise e
    return n_files_stored


def __storeAudioFile(file_id, filename, file_location, audio_files_json_filepath, **kwargs):
    """
        Save the file information in the JSON audio file.
    :param file_id: id with which the file was stored in the device (name of the file stored)
    :param filename: name associated with the file
    :param file_location: sting containing the full path of the file
    :param audio_files_json_filepath: string containing the full path of the JSON file
            that will store the information about the file provided.
    :param kwargs: Either a dictionary or a multiple key=value pairs that need to be added.
            MUST be different from the rest of the params
    :return: None

        The audio_files_json_filepath contains a list of dictionaries, each containing
        the following information regarding an audio file:
            - file_id: id with which the file was stored in the device (name of the file stored)
            - filename: name associated with the file
            - notes: some string containing a some description or notes
            - duration: string containing the duration of the audio file
            - size: size of the audio file in bytes
    """

    try:
        # create file info dictionary entry to store
        file_info = {
            "file_id": file_id,
            "filename": filename,
            "notes": "",
            "duration": getAudioFileDuration(file_location),
            "size": os.path.getsize(file_location)
        }

        if kwargs:
            log.info("Additional key-value pairs: %s", kwargs)
            for key in kwargs:
                if key not in file_info:
                    file_info[key] = kwargs[key]
        else:
            # log.info("No additional key-value pairs provided")
            pass
        # update JSON file with stored audio files information
        if os.path.isfile(audio_files_json_filepath):
            current_file_details_list = readFromJSONFile(audio_files_json_filepath)
        else:
            current_file_details_list = []
        current_file_details_list.append(file_info)
        writeToJSONFile(audio_files_json_filepath, current_file_details_list)
    except Exception as e:
        log.error(e)
        raise e


def __unzipAudioFiles(zipfile_path, destination_path, file_formats_allowed, audio_files_json_filepath):
    """
        Unzip the zip file provided and store the audio files inside in the device.
    :param zipfile_path: string containing the full path of the zip file
    :param destination_path: string containing the full path of the directory to where the files should be unzipped
    :param file_formats_allowed: list of strings containing the audio files extensions allowed
    :param audio_files_json_filepath: string containing the full path of the JSON file
            that will store the information about the audio files unzipped
    :return: number of audio files stored
        The files are unzipped, one by one, to the destination_path. For each file:
            - Validates if the device has enough space to store the audio file. If not, raises an exception.
            - Validates if the file unzipped has a valid extension. If the file is invalid,
                returns an InvalidUploadedFileError exception.
            - If everything goes well, the file is stored in the JSON audio file.
        If an exception occurs during the process, remove all the files that were unzipped so far.
    """

    files_stored = []
    try:
        with zipfile.ZipFile(zipfile_path, 'r') as fzip:
            for entry in fzip.infolist():
                filename = entry.filename
                if not filename.endswith('/'):  # ignore directories, focus only in files
                    file_id = uuid.uuid4().hex  # generate ID to be the name of the file when stored
                    destination_file_path = os.path.join(destination_path, file_id)
                    entry.filename = file_id  # store file with ID. not name
                    free_space = getStorageSize()["available"]  # get free space size in storage
                    fzip.extract(entry, destination_path)  # extract and save file in right place
                    # check if file size is larger than available free space in storage
                    # (because of 5% root dedicated space)
                    if os.stat(destination_file_path).st_size > free_space:
                        raise Exception('No space left on device')
                    checkIfFileExtensionIsAccepted(filename, file_formats_allowed)
                    __storeAudioFile(file_id, filename, destination_file_path, audio_files_json_filepath)
                    files_stored.append(destination_file_path)
    except Exception as e:
        log.error(e)
        os.remove(destination_file_path)
        removeAudioFiles(audio_files_json_filepath, files_stored)
        raise e
    else:
        try:
            os.remove(zipfile_path)
        except Exception as e:
            log.error(e)
            raise e
        else:
            return len(files_stored)


def getAudioFileDuration(file_path):
    """
        Get the duration of an audio file provided
    :param file_path: string containing the full path of the audio file
    :return: string in the format 'HH:mm:ss' representing the duration of the file
        If an error occurs during the process (for example when the file provided is not a valid audio file),
        an InvalidAudioFileError Exception is raised.
    """
    try:
        process = subprocess.Popen(['ffmpeg', '-i', file_path], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        stdout, stderr = process.communicate()
        matches = re.search(r"Duration:\s{1}(?P<duration>\d+:\d+:\d+?)(\.\d+?),", stdout.decode(), re.DOTALL).groupdict()
        file_duration = matches['duration']
    except Exception:
        log.error("{} is an invalid Audio File".format(file_path))
        raise InvalidAudioFileError
    else:
        return file_duration


def removeAudioFiles(audio_files_json_file_path, files_list):
    """
        Removes the files, present in the list provided, from the device and from the JSON file.
        The counter for deleted files is only increased when the JSON entry is deleted.
        Otherwise the user has no info of this file anyway.
    :param audio_files_json_file_path: string containing the full path of the JSON file
            that contains the information about the audio files uploaded to the device
    :param files_list: list containing the path of the files to remove
    :return: number of audio files removed
    """
    try:
        n_physical_audio_files_removed = 0
        n_audio_files_removed = 0
        if os.path.isfile(audio_files_json_file_path):
            current_audio_files_list = readFromJSONFile(audio_files_json_file_path)
        else:
            current_audio_files_list = []

        for file_path in files_list:
            if os.path.isfile(file_path):
                 # Remove the physical file
                os.remove(file_path)
                log.debug("File {} removed".format(file_path))
                n_physical_audio_files_removed += 1

            for file_info in current_audio_files_list:
                if file_path.split('/')[-1] == file_info["file_id"]:
                    current_audio_files_list.remove(file_info)
                    log.debug("File {} JSON entry removed".format(file_path))
                    n_audio_files_removed += 1
                    break

        writeToJSONFile(audio_files_json_file_path, current_audio_files_list)
        if n_physical_audio_files_removed != n_audio_files_removed:
            if n_physical_audio_files_removed < n_audio_files_removed:
                log.warning("Not all removed files had a physical counterpart! Make sure there is no device memory corruption.")
            else:
                log.warning(f"Not all physically removed files were present in the JSON! Check the integrity of the JSON file [{audio_files_json_file_path}].")

    except Exception as e:
        log.error(e)
        raise e
    else:
        return n_audio_files_removed
