import logging
import os
import string
import random
import subprocess
import uuid
from OpenSSL import crypto
from pathlib import Path
from werkzeug.utils import secure_filename

from .configuration_log import writeInConfigLog
from .uci import setUciConfigs
from .exceptions import InvalidUploadedFileError
from .constants import *
from .system_info import getStorageSize

log = logging.getLogger('flask-backend')

'''
Generate SSL Certificates to allow establishing an HTTPS connection with the WebUI
'''
def generateSSLCerts(force=False):
    # if certs dir does not exist, create it
    p = Path(SSL_CERTIFICATES_DIR)
    if not p.exists():
        try:
            p.mkdir()
        except Exception as e:
            log.error(e)
            raise e

    # if certificate do not exist, generate it
    if not force and os.path.isfile(os.path.join(SSL_CERTIFICATES_DIR, 'barix.pem')):
        log.debug("SSL certificate already exists, do not need to create it")
    else:
        try:
            # generate RSA key pair
            k = crypto.PKey()
            k.generate_key(crypto.TYPE_RSA, 2048)

            emailAddress = "info@barix.com"
            countryName = "CH"
            stateOrProvinceName = "Zurich"
            localityName = "Dubendorf"
            organizationName = "Barix AG"
            commonName = "barix.local"
            cert = crypto.X509()
            cert.get_subject().C = countryName
            cert.get_subject().ST = stateOrProvinceName
            cert.get_subject().L = localityName
            cert.get_subject().O = organizationName
            cert.get_subject().CN = commonName
            cert.get_subject().emailAddress = emailAddress
            cert.set_serial_number(uuid.uuid1().int)
            cert.gmtime_adj_notBefore(0)
            cert.gmtime_adj_notAfter(20*365*24*60*60)
            cert.set_issuer(cert.get_subject())
            cert.set_pubkey(k)
            cert.sign(k, 'sha256')

            with open(os.path.join(SSL_CERTIFICATES_DIR, 'barix.pem'), "w") as f:
                f.write(crypto.dump_privatekey(crypto.FILETYPE_PEM, k).decode("utf-8"))

            with open(os.path.join(SSL_CERTIFICATES_DIR, 'barix.pem'), "a") as f:
                f.write(crypto.dump_certificate(crypto.FILETYPE_PEM, cert).decode("utf-8"))

            # set chmod of certificate file
            os.chmod(os.path.join(SSL_CERTIFICATES_DIR, 'barix.pem'), 0o400)

            # set uci value
            setUciConfigs({'httpd.ssl.certificate':os.path.join(SSL_CERTIFICATES_DIR, 'barix.pem')}, restartServices=False)
        except Exception as e:
            log.error("Error while generating SSL certificates: {}".format(e))
            raise e


'''
Get Custom Certificates installed on the device
Returns a list of dictionaries, each representing a custom certificate. 
The information that each dictionary provides is:
    - name: name of the custom certificate
    - size: size of the custom certificate
    - state: state of the custom certificate ("Installed", ...)
'''
def getCACertificates():
    try:
        certificatesList = []
        process = subprocess.run(['custom-ca-mgr', 'status'], capture_output=True)
        output = process.stdout
        decodedOutput = output.decode("utf-8")
        lines = decodedOutput.split('\n')
        for line in lines:
            elems = line.split(' ')
            if len(elems) > 1: # last \n counts as one entire line
                #print(elems)
                filename = elems[1]
                #print(CA_CERTIFICATES_FOLDER)
                filePath = os.path.join(CA_CERTIFICATES_DIR, filename)
                #print(filePath)
                fileSize = os.path.getsize(filePath)
                state = elems[0]
                certificatesList.append({"name": filename, "size": fileSize, "state": state})
    except Exception as e:
        log.error(e)
        raise e
    else:
        log.debug("Getting CA Certificates: {}".format(certificatesList))
        return certificatesList


'''
Upload a custom certificate
@param filesUploaded: custom certificate file uploaded
Raises an InvalidUploadedFileError if Form Data didn't upload the firmware update file using the "certificate" key, or if filename is empty.
Saves the file temporarily in /tmp with a random name.
Then checks if the file size is accepted and if device has space left.
If file uploaded is too large, it will raise an InvalidUploadedFileError Exception too.
It also validates if extension is valid. At the moment, the only certificate extension allowed is ".crt". If it isn't valid, raises an InvalidUploadedFileError. 
Finally, installs the certificate and moves the file from /tmp into the final destination: CA_CERTIFICATES_DIR
If, during the process, an exception occurs, the file temporarily stored in /tmp is removed.
'''
def uploadCACertificate(filesUploaded, config_log_name=None):
    tmp_path = None
    try:
        if 'certificate' not in filesUploaded:
            raise InvalidUploadedFileError('Invalid file')
        uploadedFile = filesUploaded['certificate']
        # if user does not select file, browser also
        # submit an empty part without filename
        if uploadedFile.filename == '':
            raise InvalidUploadedFileError('Invalid filename')

        # generate random string
        letters = string.ascii_lowercase
        tmp_filename = ''.join(random.choice(letters) for i in range(8))
        tmp_path = os.path.join('/tmp', tmp_filename)

        uploadedFile.save(tmp_path)

        # check if file size is larger than available free space in storage (because of 5% root dedicated space, file is stored even if available space shows as 0, since backed is being run as root)
        if os.stat(tmp_path).st_size > getStorageSize()["available"]:
            raise Exception('No space left on device')

        # check file size
        filesize = os.stat(tmp_path).st_size
        if filesize < 133 or filesize > 16384000:
            raise InvalidUploadedFileError("File size is invalid. Please make sure you're uploading the right file.")

        # check file extension
        ALLOWED_EXTENSIONS = {'crt'}
        if '.' in uploadedFile.filename and uploadedFile.filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS:
            try:
                addCACertificates(tmp_path, secure_filename(uploadedFile.filename))
                return_msg = uploadedFile.filename + " installed."
                if config_log_name is not None:
                    writeInConfigLog(config_log_name, "Certificate " + return_msg)
                return return_msg
            except OSError:
                raise InvalidUploadedFileError('Invalid file.')
            except Exception as e:
                raise e
        else:
            raise InvalidUploadedFileError('Format not supported.')
    except Exception as e:
        if tmp_path is not None and os.path.isfile(tmp_path):
            os.remove(tmp_path)
        raise e


'''
Install a custom certificate.
@param tmp_filepath: string containing the full path of the certificate to install
@param filename: string containing the name of the custom certificate
Installs the certificate usinf openssl tool and, if successful, moves the file from it's current directory into the final destination: CA_CERTIFICATES_DIR.
custom-ca-mgr is also refreshed.
If, during the process, some error occurs and the file has already been moved to it's final destination, it is removed. 
'''
def addCACertificates(tmp_filepath, filename):
    try:
        #print('openssl x509 -in /tmp/foo')
        process = subprocess.run(['openssl', 'x509', '-in', tmp_filepath], capture_output=True)
        stderr = process.stderr
        empty = ''.encode("utf-8")
        if stderr != empty:
            raise OSError
    except Exception as e:
        log.error(e)
        raise e
    final_filepath = None
    try:
        final_filepath = os.path.join(CA_CERTIFICATES_DIR, filename)
        subprocess.run(['mv', tmp_filepath, final_filepath])
        #print('custom-ca-mgr', 'refresh')
        log.info("Certificate {} installed successfully.".format(filename))
        subprocess.run(['custom-ca-mgr', 'refresh'])
    except Exception as e:
        log.error(e)
        if final_filepath is not None and os.path.isfile(final_filepath):
            os.remove(final_filepath)
        raise e

'''
Remove custom certificates
@param certificatesList: list of certificates names to remove
Returns the number of files that were successfully removed.
For each name of the list, is there's a file in the CA_CERTIFICATES_DIR that matches the name, the file is removed.
At the end, custom-ca-mgr is refreshed.
'''
def removeCACertificates(certificatesList, config_log_name=None):
    try:
        counter = 0
        for certificateName in certificatesList:
            filePath = os.path.join(CA_CERTIFICATES_DIR, certificateName)
            if os.path.isfile(filePath):
                os.remove(filePath)
                log_msg = "Certificate {} removed"
                log.info(log_msg.format(filePath))
                if config_log_name is not None:
                    writeInConfigLog(config_log_name, log_msg.format(certificateName))
                counter += 1
        subprocess.run(['custom-ca-mgr', 'refresh'], capture_output=False)
    except Exception as e:
        log.error(e)
        raise e
    else:
        return counter


def certificateIsValid(cert_path):
    try:
        # process = subprocess.run(['openssl', 'x509', '-in', cert_path, '-noout'])
        process = subprocess.run(['openssl', 'verify', '-CAfile', cert_path, cert_path])
        if process.returncode == 0:
            return True
        else:
            return False
    except Exception as e:
        log.error(e)
        raise e


def keyIsValid(key_path):
    try:
        process = subprocess.run(['openssl', 'rsa', '-in', key_path, '-check', '-noout'])
        if process.returncode == 0:
            return True
        else:
            return False
    except Exception as e:
        log.error(e)
        raise e