import datetime
import calendar
import ifaddr
import json
import logging
import os
import psutil
import time
import shlex
import subprocess

from ipaddress import ip_address, IPv4Address
from barix import HTTPRequestsAPI

from .uci import getValueOfUci
from .utils import readFromJSONFile

log = logging.getLogger('flask-backend')


def getSystemInfo(settingsRequested):
    result_dict = {}
    try:
        for requestedSetting in settingsRequested:
            if requestedSetting == "device_type":
                # get HW_type
                dev_type = getDeviceType()
                result_dict["device_type"] = dev_type

            elif requestedSetting == "device_type_ID":
                # get HW_type_id
                dev_type_id = getDeviceTypeID()
                result_dict["device_type_ID"] = dev_type_id

            elif requestedSetting == "IPAM_type":
                # get IPAM_type
                IPAM_type = getIPAMType()
                result_dict["IPAM_type"] = IPAM_type

            elif requestedSetting == "IPAM_type_ID":
                # get IPAM_type_id
                IPAM_type_id = getIPAMTypeID()
                result_dict["IPAM_type_ID"] = IPAM_type_id

            elif requestedSetting == "mac_addr":
                # get mac address
                mac_addr = getMACaddr()
                result_dict["mac_addr"] = mac_addr


            elif requestedSetting == "reg_id":
                # get reg_id
                reg_id = getRegID()
                result_dict["reg_id"] = reg_id

            elif requestedSetting == "ip_addr":
                # get IP_addr
                ip_addr = getIPaddr('eth0')
                result_dict["ip_addr"] = ip_addr

            elif requestedSetting == "netmask":
                # get netmask
                netmask = getNetmask()
                result_dict["netmask"] = netmask

            elif requestedSetting == "default_gateway":
                # get defaultGW
                defaultGW = getDefaultGW()
                result_dict["default_gateway"] = defaultGW

            elif requestedSetting == "DSN_servers":
                # get DNS_servers
                dns_servers = getDnsServers('eth0')
                result_dict["DSN_servers"] = dns_servers

            elif requestedSetting == "FW":
                # get FW
                # TODO: get from manifest file
                result_dict["FW"] = "IP Speaker"

            elif requestedSetting == "fw_version":
                # get fwVersion
                fw_version = getFWVer()
                result_dict["fw_version"] = fw_version

            elif requestedSetting == "rootFSDate":
                rootFSDate = getRootFSCompilationDate()
                result_dict["rootFSDate"] = rootFSDate

            elif requestedSetting == "webUIVer":
                webUiVer = getWebUiVersion()
                result_dict["webUIVer"] = webUiVer

            elif requestedSetting == "app":
                # get Application
                result_dict["app"] = getApplication()

            elif requestedSetting == "app_ver":
                # get Application version
                result_dict["app_ver"] = getAppVersion()

            elif requestedSetting == "kernelVer":
                # get kernelVer
                kernelVer = getKernelVer()
                result_dict["kernelVer"] = kernelVer

            elif requestedSetting == "bootloaderVer":
                # get bootloaderVer
                bootloaderVer = getBootloaderVer()
                result_dict["bootloaderVer"] = bootloaderVer

            elif requestedSetting == "sys_time":
                # get system time
                result_dict["sys_time"] = getSystemTime()

            elif requestedSetting == "local_time":
                # get system time
                result_dict["local_time"] = getLocalTime()

            elif requestedSetting == "uptime":
                # get uptime
                result_dict["uptime"] = getUpTime()

            elif requestedSetting == "licenses":
                # get lic data
                result_dict['licenses'] = getLicenses()

            elif requestedSetting == "massStorageDev":
                # get storage devices
                result_dict["massStorageDev"] = getMassStorageDevices()

            elif requestedSetting == "HW_type_ID":
                # get HW Type ID
                result_dict["HW_type_ID"] = getHWTypeID()

            elif requestedSetting == "conn_status":
                result_dict["conn_status"] = getWiredConnStatus()

            elif requestedSetting == "poe_power_source":
                # get PoE Power Source
                result_dict["poe_power_source"] = getPoEConfig()

            elif requestedSetting == "poe_speaker_impedance":
                # get PoE Speaker Impedance
                result_dict["poe_speaker_impedance"] = getPoESpkImp()

            elif requestedSetting == "poe_amp_gain":
                # get PoE Amplifier Gain
                result_dict["poe_amp_gain"] = getPoEAmpGain()

            elif requestedSetting == "otw_status":
                # get Amplifier over temperature status
                result_dict["otw_status"] = getPoEOverTempStatus()

            elif requestedSetting == "amp_temp":
                result_dict["amp_temp"] = getAmplifierTemperature()

            elif requestedSetting == "sensor_temp":
                result_dict["sensor_temp"] = getTemperatureFromSensor()

            # @Deprecated
            elif requestedSetting == "open_source_licenses":
                result_dict["open_source_licenses"] = getOpenSourceLicenses()

            elif requestedSetting == "open_source_info":
                result_dict["open_source_info"] = {"open_source_licenses": getOpenSourceLicenses(True),
                                                   "gpl_sources_url": getGplSourcesUrl()}
            #
            # elif requestedSetting == "playerStatus":
            #     result_dict["playerStatus"] = getPlayerStatus()

            elif requestedSetting == "storage_size":
                result_dict["storage_size"] = getStorageSize()

            elif requestedSetting == "network_settings":
                result_dict["network_settings"] = getNetworkSettings()

            else:
                log.error("Unrecognized setting {}".format(requestedSetting))
    except Exception as e:
        log.error(e)
        raise e
    else:
        return result_dict


def getDeviceType():
    devTypeFile = "/tmp/factory_info_partition/factory_info_json"
    try:
        factInfoJSON = readFromJSONFile(devTypeFile)
        device_type = factInfoJSON["HW_DEVICE"]["Product_Name"]
        # log.info("Device Type: {}".format(device_type))
        return device_type
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getDeviceTypeID():
    devTypeFile = "/tmp/factory_info_partition/factory_info_json"
    try:
        factInfoJSON = readFromJSONFile(devTypeFile)
        device_type_id = factInfoJSON["HW_DEVICE"]["Product_ID"]
        # log.info("Device Type ID: {}".format(device_type_id))
        return device_type_id
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getIPAMType():
    devTypeFile = "/tmp/factory_info_partition/factory_info_json"
    try:
        factInfoJSON = readFromJSONFile(devTypeFile)
        ipamType = factInfoJSON["HW_DEVICE"]["IPAM_Name"]
        # log.info("IPAM Type: {}".format(ipamType))
        return ipamType
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getIPAMTypeID():
    devTypeFile = "/tmp/factory_info_partition/factory_info_json"
    try:
        factInfoJSON = readFromJSONFile(devTypeFile)
        ipam_type_id = factInfoJSON["HW_DEVICE"]["Legacy_IPAM_Type"]
        # log.info("Device Type ID: {}".format(ipam_type_id))
        return ipam_type_id
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getMACaddr():
    macAddrFile = "/sys/class/net/eth0/address"
    try:
        with open(macAddrFile) as file:
            mac_addr = file.read().upper().strip('\n')
            # log.info("MAC Address: {}".format(mac_addr))
            return mac_addr
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getRegID():
    configFile = "/mnt/shadow/config.json"
    try:
        configJSON = readFromJSONFile(configFile)
        regID = configJSON["registrationId"]
        return regID
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getIPaddr(ifname):
    adapters = ifaddr.get_adapters()
    for adapter in adapters:
        if adapter.nice_name == ifname:
            for ip in adapter.ips:
                try:
                    if type(ip_address(str(ip.ip))) is IPv4Address:
                        # log.info("{} ip address: {}".format(ifname, ip.ip))
                        return ip.ip
                except ValueError:
                    log.error("Invalid ip address for {}".format(ifname))
    return ''


def getNetmask():
    process = subprocess.run(['/sbin/ifconfig', 'eth0'], capture_output=True)
    output = process.stdout
    decodedOut = output.decode("utf-8")
    lines = decodedOut.split('\n')
    for line in lines:
        if line.find("netmask"):
            elems = line.split(' ')
            for idx in range(0, len(elems)):
                if elems[idx] == 'netmask':
                    return elems[idx + 1]
    return ''


def getDefaultGW():
    """cmd = ["route", "-n", "|", "grep", "eth0", "|", "grep", "'^0.0.0.0'", "|", "awk", "'{print $2}'"]
    process = subprocess.run(cmd, capture_output=True)
    output = process.stdout
    print(output)
    return output"""
    default_gw = None
    dhcp_enabled = getValueOfUci('network', 'eth0', 'proto')
    if dhcp_enabled == "dhcp":
        # if DHCP is enabled, do this way
        ps1 = subprocess.Popen(['route', '-n'], stdout=subprocess.PIPE)
        ps2 = subprocess.Popen(['grep', 'eth0'], stdin=ps1.stdout, stdout=subprocess.PIPE)
        ps3 = subprocess.Popen(['grep', '^0.0.0.0'], stdin=ps2.stdout, stdout=subprocess.PIPE)
        output = subprocess.check_output(['awk', '{print $2}'], stdin=ps3.stdout)
        ps1.wait()
        ps2.wait()
        ps3.wait()
        default_gw = output.decode("utf-8").strip('\n')
        # log.info("Default gateway: {}".format(default_gw))
    elif dhcp_enabled == "static":
        # if DHCP is disaled, do this way
        default_gw = getValueOfUci('network', 'eth0', 'gateway')
    return default_gw


def getDnsServers(ifname='eth0'):
    dns_type = getValueOfUci('network', ifname, 'dns_type')
    if dns_type == "AUTO":
        return "Auto"
    elif dns_type == "MANUAL":
        dns_servers = []
        dns1 = getValueOfUci('network', ifname, 'dns1')
        if dns1 != '' and dns1 != None:
            dns_servers.append(dns1)
        dns2 = getValueOfUci('network', ifname, 'dns2')
        if dns2 != '' and dns2 != None:
            dns_servers.append(dns2)
        return dns_servers
    else:
        return None


def getFWVer():
    fwVerFile = "/barix/info/VERSION"
    try:
        with open(fwVerFile) as file:
            fw_version = file.read().strip('\n')
            # log.info("FW Version: {}".format(fw_version))
            return fw_version
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getApplication():
    return getValueOfUci("flexa_app", "AppParam", "app_name")


def getAppVersion():
    appVerFile = "/barix/info/VERSION"
    try:
        with open(appVerFile) as file:
            app_version = file.read().strip('\n')
            return app_version
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getKernelVer():
    kernelVerFile = "/proc/version"
    try:
        with open(kernelVerFile) as file:
            kernelVer = file.read().strip('\n')
            # log.info("kernelVer: {}".format(kernelVer))
            return kernelVer
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getBootloaderVer():
    bootloaderVerFile = "/barix/info/BOOTVERSION"
    try:
        with open(bootloaderVerFile) as file:
            bootloaderVer = file.read().strip('\n')
            # log.info("bootloaderVer: {}".format(bootloaderVer))
            return bootloaderVer

    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


"""
Get device UTC time
"""


def getSystemTime():
    return time.time()


"""
Get device local time
"""


def getLocalTime():
    now = time.time()
    localTime = time.localtime(now)
    localTimestamp = calendar.timegm(localTime)
    timezoneOffset = (localTimestamp - now) / 60
    return (time.time(), timezoneOffset)


def getUpTime():
    return psutil.boot_time()


def getLicenses():
    licenses = []
    try:
        process = subprocess.run(["qiba-spi-get-license.sh"], capture_output=True)
        output = process.stdout
        decodedOutput = output.decode("utf-8")
        # log.debug(decodedOutput)
        tmp_lic_list = decodedOutput.split('\n')
        for tmp_lic in tmp_lic_list:
            if tmp_lic != '':
                data = tmp_lic.split(' ')
                # log.debug(data)
                # log.debug(len(data))
                return_dict = {"name": data[0], "status": data[5], "issueDate": data[3], "expireDate": data[4],
                               "id": data[1]}
                if data[2] == "N/A" or data[2] == "":
                    return_dict["features"] = data[2]
                else:
                    lic_features = data[2][5:-1]
                    return_dict["features"] = lic_features

                process = subprocess.run(["qiba-check-license", data[0]], capture_output=True)
                output = process.stdout

                decodedOutput = output.decode("utf-8")

                if decodedOutput.strip('\n') == data[2]:
                    return_dict["signature"] = "valid"
                else:
                    return_dict["signature"] = "invalid"

                licenses.append(return_dict)
    except Exception as e:
        log.error("Error retrieving licenses from the device: {}".format(e))
        licenses = None
    # log.info(licenses)
    return licenses


def getMassStorageDevices():
    # {"mountedDevices":[{"type":"USB","mountAs":"rw","fsType":"vfat","size":"3.7G","used":"155.1M","available":"3.6G","use":"4%","mountPoint":"/run/media/sda1"}]}
    massStorageDevicesList = []

    ps1 = subprocess.Popen(['df', '-h'], stdout=subprocess.PIPE)
    ps2 = subprocess.run(['grep', '/media/'], stdin=ps1.stdout, capture_output=True)
    ps1.wait()
    output = ps2.stdout
    # print(output)
    storageDevices = output.decode("utf-8").split('\n')
    # print(len(storageDevices))

    for line in storageDevices:
        if line != '':
            massStorageDeviceJSON = {}
            # print(line)
            dev = line.split(' ')
            paramsList = []
            for elem in dev:
                if elem != '':
                    paramsList.append(elem)
            ps2 = subprocess.Popen(['mount'], stdout=subprocess.PIPE)
            output = subprocess.check_output(['grep', paramsList[0]], stdin=ps2.stdout)
            ps2.wait()
            mountLine = output.decode("utf-8")
            mountLineDetails = mountLine.split(' ')

            if mountLineDetails[0].startswith("/dev/sd"):
                massStorageDeviceJSON["type"] = "USB"
            elif mountLineDetails[0].startswith("/dev/mmcblk"):
                massStorageDeviceJSON["type"] = "microSD"
            else:
                return {"error": "error"}

            massStorageDeviceJSON["mountAs"] = mountLineDetails[5].strip('(').split(',')[0]
            massStorageDeviceJSON["fsType"] = mountLineDetails[4]
            massStorageDeviceJSON["size"] = paramsList[1]
            massStorageDeviceJSON["used"] = paramsList[2]
            massStorageDeviceJSON["available"] = paramsList[3]
            massStorageDeviceJSON["use"] = paramsList[4].strip('%')
            massStorageDeviceJSON["mountPoint"] = paramsList[5]
            massStorageDevicesList.append(massStorageDeviceJSON)

    return massStorageDevicesList


def getHWTypeID():
    devTypeFile = "/tmp/factory_info_partition/factory_info_json"
    try:
        factInfoJSON = readFromJSONFile(devTypeFile)
        hw_type_id = factInfoJSON["HW_DEVICE"]["Legacy_HW_Type"]
        # log.info("HW Type ID: {}".format(hw_type_id))
        return hw_type_id
    except Exception as e:
        log.error("Exception found: {}".format(e))
    return ""


def getWiredConnStatus():
    f = open("/sys/class/net/eth0/operstate", 'r')
    res = f.read()
    f.close()
    return res.strip('\n')


def mapPoETypeToName(poe_type):
    if poe_type == "poe":
        return "PoE"
    elif poe_type == "poe+":
        return "PoE+"
    elif poe_type == "poe3":
        return "4PPoE"
    elif poe_type == "poe4":
        return "PoE Type 4"
    elif poe_type == "external":
        return "External"
    else:  # if uci does not exist, for example
        return "unknown"


def getPoEConfig():
    poe_type = "not available"
    uci_poe_config = getValueOfUci('poe', 'poe', 'config')
    if uci_poe_config == "auto":
        if os.path.exists('/tmp/poe-detect'):
            f = open("/tmp/poe-detect", "r")
            content = f.read()
            f.close()
            lines = content.split('\n')
            for line in lines:
                if line.startswith('poe.type'):
                    poe_type = line.split('=')[1]
                    break
            return mapPoETypeToName(poe_type)
        else:
            return "not available"
    else:
        if uci_poe_config == '' or uci_poe_config is None:
            return "not available"
        return mapPoETypeToName(uci_poe_config)


def getPoESpkImp():
    speaker_imp = "not available"
    uci_spk_imp = getValueOfUci('speaker', 'speaker', 'config')
    if uci_spk_imp == "auto":
        """
        if os.path.exists('/tmp/speaker'):
            f = open("/tmp/speaker", "r")
            content = f.read()
            f.close()
            lines = content.split('\n')
            for line in lines:
                if line.startswith('speaker.impedance'):
                    speaker_imp = line.split('=')[1]
                    break
        else:
            speaker_imp = "not available"
        """
        try:
            r = HTTPRequestsAPI.get('http://localhost:50555/api/amp/speaker_imp')
            if r.status == 200:
                info = r.read()
                enc = r.info().get_content_charset('utf-8')
                json_out = json.loads(info.decode(enc))
                speaker_imp = json_out["impedance"]
        except Exception as e:
            log.error(e)

    elif uci_spk_imp is not None and uci_spk_imp != '':
        speaker_imp = uci_spk_imp

    return speaker_imp


def getPoEAmpGain():
    amp_gain = "not available"
    try:
        r = HTTPRequestsAPI.get('http://localhost:50555/api/amp/gain')
        if r.status == 200:
            info = r.read()
            enc = r.info().get_content_charset('utf-8')
            json_out = json.loads(info.decode(enc))
            amp_gain = json_out["amp_gain"]
            if amp_gain == "0x30":
                amp_gain = "7.5V peak output voltage"
            elif amp_gain == "0x31":
                amp_gain = "15V peak output voltage"
            elif amp_gain == "0x32":
                amp_gain = "21V peak output voltage"
            elif amp_gain == "0x33":
                amp_gain = "29V peak output voltage"
            else:
                amp_gain = "unknown"
    except Exception as e:
        log.error(e)
    return amp_gain


def getPoEOverTempStatus():
    try:
        r = HTTPRequestsAPI.get('http://localhost:50555/api/amp/over_temp')
        if r.status == 200:
            info = r.read()
            enc = r.info().get_content_charset('utf-8')
            json_out = json.loads(info.decode(enc))
            over_temp = json_out["over_temp"]
            if over_temp == 1:
                return 1
    except Exception as e:
        log.error(e)
    return ""


def getAmplifierTemperature():
    try:
        r = HTTPRequestsAPI.get('http://localhost:50555/api/amp/temp')
        if r.status == 200:
            info = r.read()
            enc = r.info().get_content_charset('utf-8')
            json_out = json.loads(info.decode(enc))
            amp_temp = json_out["temperature"]
            return amp_temp
    except Exception as e:
        log.error(e)
    return ""


def getTemperatureFromSensor():
    if os.path.exists('/tmp/temp_sensor'):
        f = open("/tmp/temp_sensor", "r")
        content = f.read()
        f.close()
        return content
    return "not available"


def getAppNameFromManifest():
    manifestFilePath = '/mnt/data/package/manifest.json'
    name = ''
    try:
        if os.path.isfile(manifestFilePath):
            data = readFromJSONFile(manifestFilePath)
            if 'name' in data:
                name = data['name']
            else:
                name = "Flexa"
        else:
            name = "device"
    except Exception as e:
        log.error(e)
    return name


def getRootFSCompilationDate():
    rootFSDate = None
    rootFSDateFile = "/barix/info/DATE"
    try:
        with open(rootFSDateFile) as file:
            rootFSDate = file.read().strip('\n')
            # log.info("RootFSDate: {}".format(rootFSDate))
    except Exception as e:
        log.error("Error trying to get rootFSDate: {}".format(e))
    return rootFSDate


def getWebUiVersion():
    webUIVer = None
    webUIVerFile = "/usr/local/www/sys/VERSION"
    try:
        with open(webUIVerFile) as file:
            webUIVer = file.readline().strip('\n')
            # log.info("webUIVer: {}".format(webUIVer))
    except Exception as e:
        log.error("Error trying to get webUiVersion: {}".format(e))
    return webUIVer


def getSystemLogMessages():
    filePath = '/var/log/messages'
    nLines = 50
    return getLogLines(filePath, nLines)


def getLogLines(log_file_path, n_lines):
    lines = []
    try:
        for line in reversed(list(open(log_file_path))):
            if n_lines > 0:
                lines.append(line)
                n_lines -= 1
            else:
                break
        stringLines = ""
        for line in reversed(lines):
            stringLines += line
    except Exception as e:
        raise e
    else:
        return stringLines


def downloadAllSystemLogs():
    appName = getAppNameFromManifest()
    macAddr = getMACaddr()
    timestamp = datetime.datetime.now().strftime("%Y%m%d%H%M%S")
    filename = appName + "_" + macAddr + "_" + timestamp + "_logs.tar.gz"
    filename = filename.replace(' ', '_')

    # remove any previous log.tar.gaz that may exist
    try:
        my_dir = '/mnt/data/'  # enter the dir name
        for fname in os.listdir(my_dir):
            if fname.startswith((appName + "_" + macAddr + "_").replace(' ', '_')):
                os.remove(os.path.join(my_dir, fname))
    except Exception as e:
        log.error(e)
        raise e

    proc = subprocess.run(['uci', 'show'], capture_output=True)
    uciListToWrite = proc.stdout.decode("utf-8")
    try:
        f = open("/var/log/uci.config", "w")
        f.write(uciListToWrite)
        f.close()
    except Exception as e:
        log.error("Error creating /var/log/uci.config file: {}".format(e))
        raise e

    try:
        cmd = "tar -czf /mnt/data/" + filename + " -C /var/log ."
        subprocess.run(shlex.split(cmd))
    except Exception as e:
        log.error(e)
        raise e

    return filename


def getOpenSourceLicenses(escape_first_line=False):
    openSourceLicenses = []
    openSourceLicensesFile = "/barix/info/licenses-report.csv"
    try:
        with open(openSourceLicensesFile) as file:
            fileContent = file.read()
            fileContent = fileContent.replace('"', '')
            fileLines = fileContent.splitlines()
            for line in fileLines:
                if escape_first_line is False:
                    license = line.split(',')
                    openSourceLicenses.append(license)
                else:
                    escape_first_line = False

    except Exception as e:
        log.error("Error trying to get open source licenses: {}".format(e))
    # print(openSourceLicenses)
    return openSourceLicenses


def getGplSourcesUrl():
    content = None
    gplSourcesUrlFile = "/barix/info/GPL_SOURCES_URL"
    try:
        with open(gplSourcesUrlFile) as file:
            content = file.read()
    except Exception as e:
        log.error("Error trying to get open source URL: {}".format(e))
    return content


def getStorageSize():
    try:
        storageSizeData = {'total': '', 'used': '', 'available': ''}
        ps1 = subprocess.Popen(['df'], stdout=subprocess.PIPE)
        ps2 = subprocess.run(['grep', '/mnt/data'], stdin=ps1.stdout, capture_output=True)
        ps1.wait()
        output = ps2.stdout
        storageData = output.decode("utf-8").split('\n')

        for line in storageData:
            if line != '':
                # print(line)
                dev = line.split(' ')
                paramsList = []
                for elem in dev:
                    if elem != '':
                        paramsList.append(elem)

                if paramsList[-1].strip('\n') == '/mnt/data':
                    # storageSizeData["total"] = int(paramsList[1])
                    storageSizeData["used"] = int(paramsList[2]) * 1024
                    storageSizeData["available"] = int(paramsList[3]) * 1024
                    storageSizeData["total"] = storageSizeData["used"] + storageSizeData["available"]
                    # storageSizeData["use"] = int(paramsList[4].strip('%'))
                    storageSizeData["use"] = round(storageSizeData["used"] * 100 / storageSizeData["total"])
                    # log.debug(storageSizeData)
        return storageSizeData
    except Exception as e:
        log.error("Error getting storage size: {}".format(e))
        raise e


def getNetworkSettings():
    networkSettings = {}
    # get all network settings
    networkAdapters = ifaddr.get_adapters()
    for adapter in networkAdapters:
        if adapter.name != 'lo':
            networkSettings[adapter.name] = {}
    # log.debug(networkSettings)
    for adapter in networkSettings:
        # log.debug(adapter)
        # get connection status of network adapters
        try:
            f = open("/sys/class/net/" + adapter + "/operstate", 'r')
            status = f.read().strip('\n')
            f.close()
        except Exception as e:
            log.error(e)
            status = "unknown"
        networkSettings[adapter]["connection_status"] = status

        # log.debug(networkSettings)

        process = subprocess.run(['/sbin/ifconfig', adapter], capture_output=True)
        output = process.stdout
        lines = output.decode("utf-8").split('\n')
        for line in lines:
            # get IP Addresses of network adapters
            if line.find("inet"):
                elems = line.split(' ')
                for idx in range(0, len(elems)):
                    if elems[idx] == 'inet':
                        networkSettings[adapter]["ip_addr"] = elems[idx + 1]
            # get netmask of network adapters
            if line.find("netmask"):
                elems = line.split(' ')
                for idx in range(0, len(elems)):
                    if elems[idx] == 'netmask':
                        networkSettings[adapter]["netmask"] = elems[idx + 1]
            # if "ip_addr" in networkSettings[adapter].keys() and "netmask" in networkSettings[adapter].keys():
            #    break

        # get default gateway of network adapters
        ps1 = subprocess.Popen(['route', '-n'], stdout=subprocess.PIPE)
        ps2 = subprocess.Popen(['grep', adapter], stdin=ps1.stdout, stdout=subprocess.PIPE)
        ps3 = subprocess.Popen(['grep', '^0.0.0.0'], stdin=ps2.stdout, stdout=subprocess.PIPE)
        ps1.wait()
        ps2.wait()
        ps3.wait()
        output = subprocess.check_output(['awk', '{print $2}'], stdin=ps3.stdout)
        networkSettings[adapter]["default_gw"] = output.decode("utf-8").strip('\n')

        # get DNS servers
        networkSettings[adapter]["dns_servers"] = getDnsServers(adapter)

        # log.debug(networkSettings)

    return networkSettings
