import json
import logging
import threading
import os
from uci import Uci, UciExceptionNotFound, UciException

from .exceptions import InvalidUCIException

log = logging.getLogger('flask-backend')

'''
Get the value from a provided uci
If only config is provided, it will return a dictionary of dictionaries where:
    - first level keys are section keys and the values are dictionaries with section content.
    - second level keys are options and lists from that section. If it is an option, the value is either a string and or a tuple containing strings.
If only config and section is provided, it will return a dictionary with all options and lists in that section.
If config, section and option are provided, then it will return either a string or a tuple of strings if it consists on a list.
If the requested config or section are not found, the returned value is None.
If the requested option doesn't exist, an empty value will be returned.
If any other exception is thrown, the returned value is None.
@param config
@param section: optional
@param option: optional
'''
def getValueOfUci(config, section=None, option=None):
    value = None

    uciCli = Uci()

    # get only config
    if config is not None and section is None and option is None:
        try:
            value = uciCli.get(config)
        except UciExceptionNotFound:
            log.debug("config \"{}\" NOT FOUND".format(config))
        except Exception as e:
            log.error("Error occurred while getting config \"{}\" value: {}".format(config, e))
            raise e
    # get config and section
    elif config is not None and section is not None and option is None:
        try:
            value = uciCli.get_all(config, section)
        except UciExceptionNotFound:
            log.debug("section \"{}.{}\" NOT FOUND".format(config, section))
        except Exception as e:
            log.error("Error occurred while getting section \"{}.{}\" value: {}".format(config, section, e))
            raise e
    # get config, section and option
    elif config is not None and section is not None and option is not None:
        try:
            value = uciCli.get(config, section, option)
        except UciExceptionNotFound:
            # log.debug("uci \"{}.{}.{}\" NOT FOUND".format(config, section, option))
            value = ''
        except Exception as e:
            log.error("Error occurred while getting uci \"{}.{}.{}\" value: {}".format(config, section, option, e))
            raise e
    else:
        raise InvalidUCIException

    return value

'''
Set uci value
If option is not passed, defaults to None and the entire section value will be set. Value must be a dictionary.
If option is passed, value must be a string or a list, and option value will be set, instead of section.
If verify is True, it will check if value changed before setting. If it didn't change, it won't set the new value.
If verify is not passed, it will default to False, and the value will be set, even if it didn't changed.
uciCli is also an argument because if one uses a new uciCli for each value set before commiting, previous changes will be lost. If uciCli is none, a new one will be used.
@param config
@param section: optional
@param option: optional
@param verify: optional
@param uciCli: optional
'''
def setValueOfUci(config, section, option, value, verify=False, uciCli=None, logChangesCallback=None):
    # TODO: set only config
    # TODO: set only config and section
    canSet = True
    uciSet = False
    curr_value = None

    if uciCli == None:
        uciCli = Uci()

    value = convertBoolValueToString(value)

    # log.debug("Setting {}.{}.{}={}".format(config, section, option, value))

    if verify:
        try:
            curr_value = getValueOfUci(config, section, option)
            if isinstance(value, dict):
                if curr_value != '' and curr_value is not None:
                    canSet = False
                    curr_value_json = json.loads(curr_value)
                    for newKey in value.keys():
                        if newKey not in curr_value_json.keys():
                            canSet = True
                            break
                        else:
                            if curr_value_json[newKey] != value[newKey]:
                                canSet = True
                                break
                    for oldKey in curr_value_json.keys():
                        if oldKey not in value.keys():
                            canSet = True
                            break

            else:
                if curr_value == str(value):
                    #log.debug("Value from uci '{}.{}.{}' hasn't changed, no need to set".format(config, section, option))
                    canSet = False
        except Exception as e:
            raise e

    if canSet:
        try:
            if isinstance(value, dict):
                value = json.dumps(value)
            log.debug("setting uci '{}.{}.{}', value '{}'".format(config, section, option, value))
            uciCli.set(config, section, option, str(value))
            try:
                if logChangesCallback is not None:
                    logChangesCallback(config, section, option, curr_value, str(value))
            except Exception as e:
                log.error(e)
            uciSet = True
        except UciException: # invalid uci
            log.error("Invalid uci '{}.{}.{}'".format(config, section, option))
            raise UciException("'{}.{}.{}'".format(config, section, option))
        except Exception as e:
            log.error("Error while setting uci '{}.{}.{}' with value '{}' : {}".format(config, section, option, value, e))
            raise e

    return uciCli, uciSet

'''
Converts a python boolean value to a string.
For example, True -> "true"; False -> "false".
If value is a dict, it will iterate over it's content and convert the values of the keys if applies.
@param value: value to convert
'''

def convertBoolValueToString(value):
    if isinstance(value, bool):
        if value:
            value = "true"
        else:
            value = "false"
    return value

'''
Converts a string to the respective python value.
It can return a boolean, an integer, a string or a dictionary
@param value: string value to convert 
'''

def convertValueFromStringToTypeValue(value):
    # print("Value from beginning")
    # print(value)
    # print(type(value))
    if isinstance(value, str):
        #check if is bool
        if value == "true":
            value = True
        elif value == "false":
            value = False
        #check if is a number
        elif value.isdigit():
            value = int(value)
        else:
            #check if is dict
            try:
                valueJSON = json.loads(value)
                for elem in valueJSON:
                    # print("Key 93")
                    # print(elem)
                    valueJSON[elem] = convertValueFromStringToTypeValue(valueJSON[elem])
                    # print("Value JSON 96")
                    # print(valueJSON[elem])
                value = valueJSON
            except Exception:
                # value is string
                pass
    elif isinstance(value, dict):
        for elem in value:
            # print("Key 93")
            # print(elem)
            value[elem] = convertValueFromStringToTypeValue(value[elem])
            # print("Value JSON 106")
            # print(value[elem])
    #print("Value returned")
    #print(value)
    # bool, int and list pass right through
    return value

# def revertUci(uciTuple, uciCli=None):
#     error = ALL_OK
#
#     if uciCli == None:
#         uciCli = Uci()
#
#     try:
#         log.debug("Reverting uci {}".format(uciTuple))
#         uciCli.revert(uciTuple[0], uciTuple[1], uciTuple[2])
#     except Exception as e:
#         log.error("It was not possible to revert changes made to uci {}: {}".format(uciTuple, e))
#         error = EXCEPTION_OCCURRED_ERROR_CODE
#     return error

'''
Commit a set of uci's configs to turn permanent their value changes
@param configsList: set of configs to commit
@param uciCli: uciCli to use to commit the list of configs provided.
'''
def commitUcis(configsList, uciCli):
    for config in configsList:
        try:
            # log.debug("Commiting config {}".format(config))
            uciCli.commit(config)
        except Exception as e:
            # should never happen!!
            # return_msg = {"error_code": COMMIT_UCI_ERROR_CODE, "msg": "It was not possible to commit config {}".format(config)}
            log.error("Committing config {} failed: {}".format(config, e))
            # error code: Internal Server Error
            raise e

'''
Format a tuple containing the uci's config, section and option into a string with those parameters contatenated with '.'
For example, ('application', 'audio', 'volume') -> 'application.audio.volume'
@param uciTuple: tuple containing the uci's config, section and option
'''
def formatUciTupleIntoString(uciTuple):
    uciString = uciTuple[0]+"."+uciTuple[1]+"."+uciTuple[2]
    return uciString

'''
Get all uci's configs configured in the device.
It will get all the names of the files that exist in the directory /barix/config/current (which correspond to the configs being used in the device)
'''
def getAllConfigs():
    try:
        mypath = "/barix/config/current"
        config_names = [f for f in os.path.listdir(mypath) if os.path.isfile(os.path.join(mypath, f))]
    except Exception as e:
        log.error(e)
        raise e
    else:
        return config_names

"""
@param listOfUcis: dictionary with the list of ucis to set, where the keys are the uci's name and the value is the value to store on the uci. At the moment, only uci's names with the following sintaxe are allowed: 'config.section.option'
@param commit: (optional) flag that indicates if the changes should be commited or not. If True or not provided, changes will be commited
@param restartServices: (optional) flag that indicates if the services, that are affected by the ucis that were changed, should be restarted or not. If True or not provided, services will be restarted.
@param selectServicesToRestartCallback: (optional) callback function that selects the services that should be restarted according to the ucis that have changed. Should be provided if restartServices is True.
"""
def setUciConfigs(listOfUcis, commit=True, restartServices=True, selectServicesToRestartCallback=None, logChangesCallback=None):
    # log.info("Ucis to set: {}".format(listOfUcis))
    ucisConfigured = []  # in case of failure, to revert
    configsToCommit = set()  # to commit ucis
    uciCli = None
    try:
        for uci in listOfUcis:
            params = uci.split('.')
            if len(params) == 3:
                # config = params[0], section = params[1], option = params[2], value = jsonUcis[uci]
                uciCli, uciSet = setValueOfUci(params[0], params[1], params[2], listOfUcis[uci], True, uciCli, logChangesCallback)
                if uciSet:
                    ucisConfigured.append((params[0], params[1], params[2]))
                    configsToCommit.add(params[0])

        # log.info("Ucis set: {}".format(ucisConfigured))

        # commit uci changes?
        if commit:
            # uci changes must be committed
            commitUcis(configsToCommit, uciCli)

        # restart services?
        if restartServices:
            if selectServicesToRestartCallback is not None:
                restartRelatedServices(ucisConfigured, selectServicesToRestartCallback)
            else:
                log.error("selectServicesToRestartCallback is None")
                raise Exception

    except UciException as e:
        log.error(e)
        raise e
    except Exception as e:
        log.error("Error type [%s] appeared in [%s] when executing line [%s]",
                        type(e).__name__, __file__, e.__traceback__.tb_lineno)
        log.error(e)
        raise e
    else:
        return ucisConfigured

'''
Calls the callback function to create a list of services that should be restarted according to the ucis that were changed.
Write the list into the file that stores the services to restart. 
If the file existed already, the new list will be merged with the existing one and duplicates will be removed.
Then, calls restartServicesFromFile to restart the services on a separate thread, according to the order of priority.
@param ucisConfigured: list(array) of tuples containing the ucis that were configured. At the moment, each tuple contains a uci's config, a uci's section and a uci's option.
@param selectServicesToRestartCallback: callback function that selects the services that should be restarted according to the ucis that have changed
'''
def restartRelatedServices(ucisConfigured, selectServicesToRestartCallback):
    try:
        from .restart_services import readFromRestartServicesFile, writeIntoRestartServicesFile, restartServicesFromFile
        servicesList = selectServicesToRestartCallback(ucisConfigured)
        # write servicesList in file
        if len(servicesList) > 0:
            # read services from file
            servicesFromFile = readFromRestartServicesFile()
            # join services from list and from file
            listOfServices = []
            for service in servicesFromFile:
                listOfServices.append(service)
            for service in servicesList:
                listOfServices.append(service)
            # remove duplicates
            setOfServices = set(listOfServices)
            log.debug("List of services to write into file: {}".format(setOfServices))
            writeIntoRestartServicesFile(setOfServices)
        else:
            log.info("No services to restart")

        # restart services
        t = threading.Thread(target=restartServicesFromFile, args=())
        t.start()
    except Exception as e:
        log.error(e)
        raise e
