#!/usr/bin/lua
local logging = require("logging");
--local log_console = require("logging.console");
local log_file = require("logging.rolling_file");

local logger = log_file{
    filename="/var/log/arcanix.log",
    maxFileSize=2048*1024,
    maxBackupIndex=3,
    logLevel=logging.DEBUG,
    logPattern="%date %level %file:%line %message\n"
};

--[[
local logger = log_console{
    timestampPattern="%Y-%m-%d %X",
    logPattern="%date %level %file:%line %message\n"
};
]]

-- always set default logger before load arcanix module
logging.defaultLogger(logger);

local uv = require("luv");
local arcanix = require("arcanix");
local uci_handler = require("arcanix.uci_handler")

local RELOAD_SETTINGS_URL = "http://127.0.0.1:48081/api/v1/reload-settings";
local IDENTIFY_XD_URL = "http://127.0.0.1:48081/api/v1/identify-xd";
local AMP_POWER_SWITCH = "http://127.0.0.1:48081/api/v1/switch-amp";

local function finish_app()
    arcanix.stop()
    os.exit(0)
end

local function readModel(filename)
    local f = io.open(filename, "rb")
    if f then
        local content = f:read("*a")
        f:close()
        return content
    end

    return nil
end

local term_signal = uv.new_signal()
term_signal:start("sigint", finish_app)
term_signal:start("sigterm", finish_app)

local http_work = uv.new_work(
    function(url)
        local ll = require("logging");
        local logger = ll.defaultLogger();
        logger:info("sending http request to %s", url);

        http = require("socket.http");
        http.TIMEOUT = 5000;

        local request_body = "";
        local response_body = {};
        local ltn12 = require("ltn12")

        local res, code, response_headers = http.request{
          url = url,
          method = "PUT", 
          headers = 
            {
                ["Content-Length"] = #request_body;
            },
            source = ltn12.source.string(request_body),
            sink = ltn12.sink.table(response_body),
        }

        return code;
    end,
    function(code)
        logger:debug("Return code %s", code);
    end
);

local function reloadSettings()
    http_work:queue(RELOAD_SETTINGS_URL);
end

local shell_work = uv.new_work(
    function(cmd)
        local ll = require("logging");
        local logger = ll.defaultLogger();
        logger:info("executing shell cmd %s", cmd);
        result = os.execute(cmd);
        reloadSettings();
        return result;
    end,
    function(ret)
        logger:info("Shell returned %s", ret);
    end
);

local function applyNetworkSettings()
    shell_work:queue("/etc/init.d/networking restart");
end

local function rebootDevice()
    shell_work:queue("/sbin/reboot");
end

local function applySNMP()
    shell_work:queue("/barix/apps/epic-sip/applySNMP.sh");
end

local uci_map = {
    ["network.eth0.proto"] =         {uuid=0x00010000, cmd=applyNetworkSettings},
    ["network.eth0.ipaddr"] =        {uuid=0x00010001, cmd=applyNetworkSettings},
    ["network.eth0.netmask"] =       {uuid=0x00010002, cmd=applyNetworkSettings},
    ["network.eth0.gateway"] =       {uuid=0x00010003, cmd=applyNetworkSettings},
    ["network.eth0.dns_type"] =      {uuid=0x00010004, cmd=applyNetworkSettings},
    ["network.eth0.dns1"] =          {uuid=0x00010005, cmd=applyNetworkSettings},
    ["network.eth0.dns2"] =          {uuid=0x00010006, cmd=applyNetworkSettings},
    
    ["epic.server.ipaddr"] =         {uuid=0x00010007, cmd=reloadSettings},
    ["epic.snmp.enable"] =           {uuid=0x00010036, cmd=applySNMP},
    
    ["ae_ms.mixer.ch1"] =            {uuid=0x00010009, cmd=reloadSettings},
    ["ae_ms.mixer.ch4"] =            {uuid=0x0001000C, cmd=reloadSettings},
    ["ae_ms.mixer.ch5"] =            {uuid=0x0001000D, cmd=reloadSettings},
    ["ae_ms.mixer.ch100"] =          {uuid=0x0001000F, cmd=reloadSettings},
    ["ae_ms.mixer.enable1"] =        {uuid=0x00010026, cmd=reloadSettings},
    ["ae_ms.mixer.enable4"] =        {uuid=0x00010029, cmd=reloadSettings},
    ["ae_ms.mixer.enable5"] =        {uuid=0x0001002A, cmd=reloadSettings},
    ["ae_ms.mixer.enable100"] =      {uuid=0x0001002C, cmd=reloadSettings},
    
    ["ae_ms.volume.out1"] =          {uuid=0x00010010, cmd=reloadSettings},
    ["ae_ms.volume.out4"] =          {uuid=0x00010013, cmd=reloadSettings},
    ["ae_ms.volume.enable1"] =       {uuid=0x0001002E, cmd=reloadSettings},
    ["ae_ms.volume.enable4"] =       {uuid=0x00010031, cmd=reloadSettings},

    ["ae_ms.volume.out0"] =          {uuid=0x00010025, cmd=reloadSettings},
    ["ae_ms.volume.enable0"] =       {uuid=0x0001002D, cmd=reloadSettings},
    
    ["ae_ms.serial.config_1"] =      {uuid=0x00010020, cmd=reloadSettings},
    ["ae_ms.serial.config_5"] =      {uuid=0x00010024, cmd=reloadSettings},
    
    ["ae_ms.serial.egress_list_1"] = {uuid=0x0001001A, cmd=reloadSettings},
    ["ae_ms.serial.egress_list_5"] = {uuid=0x0001001E, cmd=reloadSettings},
    ["ae_ms.serial.egress_list_6"] = {uuid=0x0001001F, cmd=reloadSettings}
};

local handler = uci_handler:new()
local model = readModel("/barix/apps/epic-sip/MODEL")

for uci, param in pairs(uci_map) do
    handler:add_configuration(uci, param.uuid, param.cmd);
end

-- wait for the dongle to be ready
uv.sleep(30000);

arcanix.start(uv, arg[1], handler, function(value)
    if value == "beep-tone" then
        os.execute("aplay beep.wav");
    elseif value == "blink-leds" then
        os.execute("/barix/apps/epic-sip/ble_connected.sh");
    elseif value == "xd-leds" then
        http_work:queue(IDENTIFY_XD_URL);
    elseif value == "switch-amp" then
        http_work:queue(AMP_POWER_SWITCH);
    end
end, model);

while true do
    uv.run();
end
