import socket
import time
import random
import logging
import threading
import shutil
import os.path
import subprocess
from urllib.error import URLError

from barix import HttpMethod

class FwUpdater:
    
    def __init__(self,updateUrl:str, stateFilePath: str):
        self.updateUrl      = updateUrl
        self.stateFilePath  = stateFilePath
        self.lastCheck      = None
        self.updateTime     = None
        self.iniWindowTime  = -1
        self.lock           = threading.Lock()
        self.updateRunning  = False
        self.timer          = None
        self.baseHr         = None
        self.baseMin        = None
        self.timeWindowLen  = None
        self.logger         = logging.getLogger("fw-updater")
        if os.path.isfile(stateFilePath):
            with open(stateFilePath,"r") as f:
                self.curFwLastModified = f.read()
        else:
            self.curFwLastModified = ""

    def setIniWindowTime(self,secs : int ) -> None:
        self.iniWindowTime  = time.time() + secs

    def setUpdateBaseTime(self, hr: int, min: int):
        self.baseHr     = hr
        self.baseMin    = min


    def setUpdateWindowDuration(self,duration: int):
        self.timeWindowLen  = duration


    def calculateUpdateTime(self):
        """
        Calculate a somewhat random hour between
        base and base+window_duration
        """
        randNum = random.randint(0,self.timeWindowLen)
        hrRand  = self.baseHr +  randNum//60
        mnRand  = self.baseMin + randNum%50
        self.updateTime = int(hrRand*10000 + mnRand*100)
        self.logger.info("next update set for {:0>6d}".format(self.updateTime))

    def checkFwUpdates(self):
        """
        Basic check for firmware updates
         * performs HEAD to the URL and compares the Last Modified
           with the currently stored version.
         * if timestamps are found to differ, the download is
           performed and the file is installed.
        """
        secs        = time.time()
        if secs<=self.iniWindowTime and not self.updateRunning:
            self.logger.info("Time within initial update window. spawning updater.")
            self.spawnUpdaterThread()
            return

        if self.updateTime is None:
            self.logger.info("update time is not set. Calculating it...")
            self.calculateUpdateTime()

        now         = time.localtime()
        timestamp   = int(now.tm_hour*10000 + now.tm_min*100 + now.tm_sec)
        
        if self.lastCheck is not None:
            if self.lastCheck<=self.updateTime and timestamp>self.updateTime and not self.updateRunning:
                self.spawnUpdaterThread()
        
        self.lastCheck  = timestamp
        #self.logger.debug("fw update last checked at {:0>6d}".format(self.lastCheck))

    
    def checkAndRunUpdate(self) -> bool:
        try:
            self.logger.debug("checking URL: '{}'".format(self.updateUrl))
            response    = HttpMethod.head(self.updateUrl)
            if response.status!=200:
                self.logger.error("Cannot check firmware. HEAD returned {}".format(response.status))
                return False

            rxLastModified  = response.headers.get("last-modified")
            if rxLastModified is None:
                self.logger.warning("update server doesn't provide 'last-modified'. Updating just in case...")
                return self.runUpdate()

            if rxLastModified!=self.curFwLastModified:
                self.logger.info("fw 'last-modified': '{}' --> '{}'. updating".format(self.curFwLastModified,rxLastModified))
                return self.runUpdate()
            else:
                self.logger.info("no change detected: last modified is '{}'".format(self.curFwLastModified))
                return True
        except URLError as err:
            reason = err.reason
            if type(reason) is str:
                self.logger.info("URL error downloading update: {}".format(reason))
            elif type(reason) is socket.timeout:
                self.logger.info("timeout checking update")
            else:
                self.logger.info("URL error downloading update:", exc_info=err)
            return False
        except Exception as ex:
            self.logger.warning("Update finished with exception: ", exc_info=ex)
            return False


    def spawnUpdaterThread(self):
        with self.lock:
            try:
                if self.updateRunning:
                    self.logger.warning("Updater thread is already running.")
                    return
                else:
                    self.updateRunning  = True
                    self.logger.info("launching firmware update task...")
                    self.timer  = threading.Timer(0.5,self.updaterWorkerFunction)
                    self.timer.start()
            except Exception as ex:
                self.logger.error("Oops: some exception while spawning updater thread.",exc_info=ex)

    def updaterWorkerFunction(self):
        maxAttempts = 5
        for attempt in range(0,5):
            self.logger.debug("running attempt {}/{}".format(attempt+1,maxAttempts))
            if self.checkAndRunUpdate():
                self.logger.debug("sucessful check. exit.")
                with self.lock:
                    self.updateRunning  = False
                    return
            else:
                self.logger.warning("update run failed. retrying in 10 seconds...")
                time.sleep(10)
        
        self.logger.error("gave up running update after {} failed attempts".format(maxAttempts))
        with self.lock:
            self.updateRunning  = False
            return

    def runUpdate(self) -> bool:
        rxLastModified  = ""
        with open("/mnt/data/fw_update_tmp.tar","wb") as f:
            self.logger.debug("getting update from '{}'".format(self.updateUrl))
            try:
                resp    = HttpMethod.get(self.updateUrl)
                self.logger.info("http get completed")
            except Exception as ex:
                self.logger.error("Exception running download: ",exc_info=ex)
                return False

            
            try:
                if resp.status!=200:
                    self.logger.error("failed to fetch update with HTTP Status {}".format(resp.status))
                    return False

                self.logger.info("copying into destination")
                shutil.copyfileobj(resp,f)
                self.logger.info("copy finished")
                if resp.headers.get("last-modified") is not None:
                    rxLastModified  = resp.headers.get("last-modified")
                else:
                    self.logger.warning("last-modified header is missing")
            except URLError as err:
                reason = err.reason
                if type(reason) is str:
                    self.logger.error("URL error downloading update: {}".format(reason))
                elif type(reason) is socket.timeout:
                    self.logger.error("timeout downloading update")
                else:
                    self.logger.error("URL error downloading update:",exc_info=reason)
                return False
            except Exception as ex:
                self.logger.error("Exception handling response: ", exc_info=ex)
                return False

        try:
            self.logger.info("running the updater...")
            ph = subprocess.Popen(["qiba-update-client","-f","/mnt/data/fw_update_tmp.tar"])
            ret= ph.wait()
            ret = 0
            if ret==0:
                self.logger.info("update install successful. saving timestamp and rebooting...")
                with open(self.stateFilePath,"w") as f:
                    f.write(rxLastModified)

                subprocess.Popen("reboot")
                return True
            else:
                self.logger.info("update failed with code {}".format(ret))
                return False
        except Exception as ex:
            self.logger.error("Exception running updater: ", exc_info=ex)
            return False
        


