#!/usr/bin/env python3
import math
from os import statvfs_result
import os.path
import threading
from types import ClassMethodDescriptorType
from urllib.parse import urlparse
import time
import signal
import logging
from logging.handlers import RotatingFileHandler
import syslog

from serial.serialposix import Serial
import alsaaudio
import sys
from uci import Uci
from typing import *
import random
import json
import traceback
import subprocess

from barix.media.RtpAudioSource import *
from barix.media.SipClientAudioSource import *
from barix.io.Gpio import Gpio
from barix.io import leds
from barix.licensing import *
from barix.web.system_info import getMACaddr
import baco

from http_api import HttpApi
from playback_manager import *
from fw_updater import FwUpdater
from tcp_api import TcpApi
from serial_gw import SerialGw

from i2c import I2C
from padauk import Padauk

isCZ300 = False
PIN1 = 0
PIN2 = 1
PIN3 = 2
PIN4 = 3
PIN5 = 4
PIN6 = 5
PIN7 = 6
PIN8 = 7
PIN9 = 8
PIN10 = 9
PIN11 = 10
PIN12 = 11
PIN13 = 12
PIN14 = 13
PIN15 = 14
PIN16 = 15
# GPIO_CZ300=(412, 413, 414, 415, 408, 409, 410, 411, 400, 401, 0, 0, 0, 0, 0, 407)     # v1.0 of the hardware
GPIO_CZ300 = (408, 409, 410, 411, 412, 413, 414, 415, 400, 401, 402, 403, 404, 405, 406, 407)

LIMITTED_MODELS=["MS-375", "MS-775", "MS-720"]

# Gamma correction for the pots
GAMMA=15
INVGAMMA=1/GAMMA

VOL_THRSHLD = -79       # db

# Audio input indexes
AUDIO_INPUTS = [1, 4, 5, 6, 100]
AUDIO_OUTPUTS= [0, 5, 0, 0, 6]

class InOut:
    IDLE = 0
    PRESSED = 1
    RELEASED = 2

    def __init__(self, wpa_io: int):
        pin = GPIO_CZ300[wpa_io]

        self.lock = threading.Lock()
        self.mode = Gpio.IN
        self.outVal = None
        self.gpio = Gpio(str(pin), "in")
        self.state = self.gpio.read()
        self.relayOn = False

    def checkEvents(self) -> int:
        """
        Check for updates on the input
        """
        try:
            self.lock.acquire()
            if self.relayOn:
                return InOut.IDLE

            stt = self.gpio.read()
            if self.state == stt:
                return InOut.IDLE

            self.state = stt
            if stt == 1:
                return InOut.RELEASED
            return InOut.PRESSED
        finally:
            self.lock.release()

    def read(self) -> int:
        """
        Read the input. Does not update event tracking.
        """
        try:
            self.lock.acquire()
            if self.relayOn:
                return 1

            return self.gpio.read()
        finally:
            self.lock.release()

    def setRelay(self, value: bool) -> None:
        try:
            self.lock.acquire()
            if value:
                if not self.relayOn:
                    self.gpio.setDirection(Gpio.OUT)
                    self.gpio.write(0)
                    self.relayOn = True
            else:
                self.gpio.setDirection(Gpio.IN)
                self.relayOn = False
        finally:
            self.lock.release()

    def getMode(self) -> int:
        if self.relayOn:
            return 1
        return 0

def ipMap(ip):
    return [0, 3, 4, 1, 2][ip]

def opMap(op):
    return [0, 3, 4, 1, 2, 5, 6, 7, 8, 0, 0, 0, 0, 13][op]

def uciGet(uci: Uci, config: str, section: str, opt: str) -> str:
    try:
        return uci.get(config, section, opt)
    except:
        return None


def uciSet(uci: Uci, config: str, section: str, opt: str, value: str) -> None:
    try:
        uci.set(config, section, opt, value)
        uci.commit(config)
    except:
        pass


def decodeDCommand(line: str) -> Tuple[str, str]:
    if len(line) < 2:
        return ("", None)

    if line == "DS":
        return ("DS", None)
    elif line == "DA":
        return ("DA", None)
    else:
        # Dsip://id@server
        # DHext
        # DHsip://id@server
        # Dext
        if line.find("DH") == 0:
            if len(line) > 3:
                return ("DH", line[2:])
            else:
                # DH without arguments makes no sense!!
                return ("", None)
        else:
            return ("D", line[1:])


class EpicSip:
    ledR = Gpio("10", "out")
    ledR.setDirection(Gpio.OUT)
    ledG = Gpio("13", "out")
    ledG.setDirection(Gpio.OUT)

    HTTP_API_PORT = 48081

    sipDict = {
        SipCallState.IDLE: "IDLE",
        SipCallState.CALLING: "CALLING",
        SipCallState.INCOMMING_CALL: "INCOMMING",
        SipCallState.RINGING: "RINGING",
        SipCallState.CALLING_RINGING: "RINGING",
        SipCallState.CONNECTED: "CONNECTED",
        SipCallState.GETTING_NOTIF: "GETTING_NOTIF"
    }

    padauk1=Padauk(1)
    padauk2=Padauk(2)

    channelVolume = [0, 0, 0, 0, 0, 0, 0]
    channelEnabled = [True, True, True, True, True, True, True]
    preSXIstate = [True, True, True, True, True]  # Normal (before SXI command) state of input channels.
    inPotLUT=[]
    outPotLUT=[]

    serialConfig = [None, None]
    gwForward1 = 0
    gwForward2 = 0

    _buttonStatus = 0x0F
    _buttonPStatus = 0x0F

    fwRP = None
    fwXD = None
    fwTCP = None

    xdActive = False

    def __init__(self):
        self.logger = logging.getLogger("epic")
        with open("/barix/info/VERSION", "r") as fIn:
            self.sysVersion = fIn.readline().strip()

        self.ipPotValue=-1
        self.opPotValue=-1

        with open("/barix/apps/epic-sip/MODEL") as fIn:
            self.model = fIn.readline().strip()
            if self.model in LIMITTED_MODELS:
                global AUDIO_INPUTS
                AUDIO_INPUTS = [1, 4, 6, 100]
                self.ipPotValue=1
                self.opPotValue=1
        print("MODEL: {}".format(self.model))
        print(self.model not in LIMITTED_MODELS)
        
        self.ipPotPrevVal=-1  # Initialize to value outside the working space (0 - 255)
        self.opPotPrevVal=-1  # Initialize to value outside the working space (0 - 255)

        self.i2c=I2C()

        syslog.openlog("AE-MSx00")
        msg = "CZ-300/301 controller starting. model={} sysVersion={}".format(self.model, self.sysVersion)
        self.logger.info(msg)
        syslog.syslog(msg)
        self.stopApplication = False
        signal.signal(signal.SIGTERM, self.signalHandler)
        signal.signal(signal.SIGINT, self.signalHandler)

    def potRange(self, pot, dbRange) -> int:
        return int(pot/255*dbRange)

    def buildInPotLUT(self):
        self.inPotLUT=[]
        dbRange=self.channelVolume[3]-VOL_THRSHLD
#        print("{} {} -> {}".format(self.channelVolume[4], VOL_THRSHLD, dbRange))
        for n in range(0, 256):
            v=int(((n*0.00392)**INVGAMMA)*255)
            k=self.potRange(v, dbRange)
            m=int(VOL_THRSHLD+k)
            self.inPotLUT.append(m+1)
#            print("{} ({}) {} -> {}".format(n, v, k, m))

    def buildOutPotLUT(self):
        self.outPotLUT=[]
        dbRange=self.channelVolume[6]-VOL_THRSHLD
#        print("{} {} -> {}".format(self.channelVolume[4], VOL_THRSHLD, dbRange))
        for n in range(0, 256):
            v=int(((n*0.00392)**INVGAMMA)*255)
            k=self.potRange(v, dbRange)
            m=int(VOL_THRSHLD+k)
            self.outPotLUT.append(m+1)
#            print("{} ({}) {} -> {}".format(n, v, k, m))

    def reloadSettings(self):
        uci = Uci()

        self.buildInPotLUT()
        self.buildOutPotLUT()

        # These are the input channels
        for ch in [1, 4, 5, 100]:
            chi = self.mapInputChannel(ch)
            self.channelVolume[chi] = int(uciGet(uci, "ae_ms", "mixer", "ch{}".format(ch)))
            en = True if uciGet(uci, "ae_ms", "mixer", "enable{}".format(ch)) == "true" else False
            self.channelEnabled[chi] = en
            self.preSXIstate[chi] = en
            self.setChannelVolume(chi)

        # These are the output channels (output channels are channels 5 and 6 on the mixer IC)
        for ch in [0, 1, 4]:
            chi = self.mapOutputChannel(ch)
            if chi==0:      #Invalid channel #
                continue
            else:
                self.channelVolume[chi] = int(uciGet(uci, "ae_ms", "volume", "out{}".format(ch)))
                self.channelEnabled[chi] = True if uciGet(uci, "ae_ms", "volume", "enable{}".format(ch)) == "true" else False
                self.setOutVolume(chi)

        self.buildInPotLUT()
        self.buildOutPotLUT()

        self.serialConfig[0] = uciGet(uci, "ae_ms", "serial", "config_1")
        self.serialConfig[1] = uciGet(uci, "ae_ms", "serial", "config_5")

        val     = uciGet(uci, "ae_ms", "talkback", "level_wpa")
        gain    = -79
        if len(val)>0:
            try:
                gain    = int(val)
                gain = max(-79, min(16, gain))
                gain = int((gain + 79) * 2.85)
                cmd="amixer -D hw:2 cset name='WPA Capture Volume' '" + str(gain) + "'"
                p = subprocess.Popen(cmd, shell=True)
                p.wait()
            except:
                self._logger.error("invalid ae_ms.talkback.level_wpa: '{}'".format(val))

        val     = uciGet(uci, "ae_ms", "talkback", "level_rp1")
        gain    = -79
        if len(val)>0:
            try:
                gain    = int(val)
                gain = max(-79, min(16, gain))
                gain = int((gain + 79) * 2.85)
                cmd="amixer -D hw:2 cset name='RP1 Capture Volume' '" + str(gain) + "'"
                p = subprocess.Popen(cmd, shell=True)
                p.wait()
            except:
                self._logger.error("invalid ae_ms.talkback.level_rp1: '{}'".format(val))

    # For my Sanity can we follow, as much as possible the format from the 700.  My old brain will hurt of we have 2 different audio layouts.
    # Inputs
    # RP is 1
    # Line input 2
    # 3 nada  (not sure if this is the correct thing to do) adding @Jayme Wright and @Zack Loder for input on this.
    # XD is 4
    # Network(Barix) is 00

    # Outputs
    # Output to the amplifier is 0
    # RP1 is 1
    # Output 2 the 3.5 line out

    def mapInputChannel(self, ch: int):
        if ch == 100:
            ch = 6
        if ch < 1 or ch > 6:
            return 0

        if self.model in LIMITTED_MODELS:
            return (0, 4, 0, 0, 1, 0, 2)[ch]
        return (0, 4, 0, 0, 1, 3, 2)[ch]

    def mapRevChannel(self, ch: int):
        if ch>4 or ch<0:
            return 0
        if self.model in LIMITTED_MODELS:
            return (0, 4, 6, 0, 1, 0, 0)[ch]
        return (0, 4, 6, 5, 1, 0, 0)[ch]

    def mapOutputChannel(self, ch: int):
        if ch>4 or ch<0:
            return 0
        if self.model in LIMITTED_MODELS:
            return (6, 5, 0, 0, 0)[ch]
        return (0, 5, 0, 0, 6)[ch]

    def ledShowoff(self):
        colors=['red', 'green', 'blue', 'light blue', 'white']
        for led in range(1, 5):
            for color in colors:
                self.padauk1.lightLED(led, 3, color)
                time.sleep(0.1)

        for led in [4, 3, 2, 1]:
            for i in [4, 3, 2, 1, 0]:
                self.padauk1.lightLED(led, i, "white")
                time.sleep(0.1)

    def updateLED(self, led):
        rState=0
        if led==opMap(3):
            rState=self.io5.getMode()
        elif led==opMap(4):
            rState=self.io6.getMode()
        elif led==opMap(1):
            rState=self.io7.getMode()
        elif led==opMap(2):
            rState=self.io8.getMode()

        iState=1
        if led==ipMap(1):
            iState=self.io1.read()
        elif led==ipMap(2):
            iState=self.io2.read()
        elif led==ipMap(3):
            iState=self.io3.read()
        elif led==ipMap(4):
            iState=self.io4.read()

        if rState==1:
            if iState==0:
                self.padauk1.lightLED(led, 4, "white")
            else:
                self.padauk1.lightLED(led, 4, "blue")
        else:
            if iState==0:
                self.padauk1.lightLED(led, 4, "yellow")
            else:
                self.padauk1.lightLED(led, 4, "off")

    def setRelay1(self, stt: bool) -> None:
        """
        Sets the state of relay1 to ON(True) or OFF(False)

        Returns true is state changed, which requires a status
        update to be sent through TCP connected clients
        """
        sttStr = "OFF"
        if stt:
            sttStr = "ON"

        syslog.syslog("RELAY1 --> {}".format(sttStr))
        self.io7.setRelay(stt)
        self.updateLED(opMap(1))

    def setRelay2(self, stt: bool) -> None:
        """
        Sets the state of relay2 to ON(True) or OFF(False)

        Returns true is state changed, which requires a status
        update to be sent through TCP connected clients
        """
        sttStr = "OFF"
        if stt:
            sttStr = "ON"
        syslog.syslog("RELAY2 --> {}".format(sttStr))
        self.io8.setRelay(stt)
        self.updateLED(opMap(2))

    def setRelay3(self, stt: bool) -> None:
        """
        Sets the state of relay3 to ON(True) or OFF(False)

        Returns true is state changed, which requires a status
        update to be sent through TCP connected clients
        """
        sttStr = "OFF"
        if stt:
            sttStr = "ON"
        syslog.syslog("RELAY3 --> {}".format(sttStr))
        self.io5.setRelay(stt)
        self.updateLED(opMap(3))

    def setRelay4(self, stt: bool) -> None:
        """
        Sets the state of relay4 to ON(True) or OFF(False)

        Returns true is state changed, which requires a status
        update to be sent through TCP connected clients
        """
        sttStr = "OFF"
        if stt:
            sttStr = "ON"
        syslog.syslog("RELAY4 --> {}".format(sttStr))
        self.io6.setRelay(stt)
        self.updateLED(opMap(4))

    def setLed2(self, stt: int) -> None:
        """
        Sets the state of LED2 to Red(1), Green(2) or OFF(0)

        Returns true is state changed, which requires a status
        update to be sent through TCP connected clients
        """
        sttStr="UNKNOWN STATE"
        if stt==1:
            self.ledR.write(0)
            self.ledG.write(1)
            sttStr="RED"
        elif stt==2:
            self.ledR.write(1)
            self.ledG.write(0)
            sttStr="GREEN"
        elif stt==0:
            self.ledR.write(1)
            self.ledG.write(1)
            sttStr="OFF"
        syslog.syslog("LED2 --> {}".format(sttStr))

    def signalHandler(self, sigNum: int, frame):
        msg = "Caught signal {}. signaling shutdown".format(sigNum)
        self.logger.info(msg)
        self.stopApplication = True

    def bringUpMixers(self):
        self.logger.info("bringing up mixer channels")
        alsaaudio.PCM(device='channel1')
        alsaaudio.PCM(device='channel2')
        alsaaudio.PCM(device='channel3')
        alsaaudio.PCM(device='channel4')
        alsaaudio.PCM(device='channel5')

    # Channel# is the non-mapped channel#
    def storeChannelConfig(self, ch: int, chi: int):
        uci = Uci()
        if ch == 6:
            ch = 100
        if chi in [1, 2, 3, 4, 100]: # Input channels
            self.buildInPotLUT()
            option = "ch{}".format(ch)
            uciSet(uci, "ae_ms", "mixer", option, str(self.channelVolume[chi]))
            option = "enable{}".format(ch)
            uciSet(uci, "ae_ms", "mixer", option, str(self.channelEnabled[chi]).lower())
            self.setChannelVolume(chi)
        else: # (5, 6)
            self.buildOutPotLUT()
            option = "out{}".format(ch)
            uciSet(uci, "ae_ms", "volume", option, str(self.channelVolume[chi]))
            option = "enable{}".format(ch)
            uciSet(uci, "ae_ms", "volume", option, str(self.channelEnabled[chi]).lower())
            self.setOutVolume(chi)

    # Channel# is the internal, mapped channel# (1, 2, 3, 4)
    def setChannelVolume(self, ch: int):
        if ch<1 or ch>5:
            return

        vol=-79
        if self.channelEnabled[ch]:
            vol =self.channelVolume[ch]
            if self.model not in LIMITTED_MODELS:
                if ch==3:
                    if self.ipPotValue<0:
                        return
                    vol = self.ipPotValue
                    vol = self.inPotLUT[vol]
                    print("{} {}".format(self.ipPotValue, vol))
                if ch==6:
                    vol = self.opPotValue
                    vol=self.outPotLUT[vol]
            if vol < -79:
                vol = -79

        # print("POT_IN: {} VOL: {}".format(ch, vol))
        self.i2c.setChannelVolume(ch, vol)

    # Channel# is the internal, mapped channel# (5, 6)
    def setOutVolume(self, ch: int):
        if self.channelEnabled[ch]:
            vol=self.channelVolume[ch]
            if ch==6 and self.model not in LIMITTED_MODELS:
                vol = self.outPotLUT[self.opPotValue]
                
            # print("POT_OUT: {} VOL: {}".format(ch, vol))
            self.i2c.setChannelVolume(ch, vol)
        else:
            self.i2c.setChannelVolume(ch, -79)

    def restoreChannelState(self):
        for ch in (1, 2, 3, 4):
            self.channelEnabled[ch] = self.preSXIstate[ch]
            self.setChannelVolume(ch)

    def setMixerCParam(self, param, value):
        p = subprocess.Popen("amixer -D hw:2 cset " + "name=\'" + param + "\'" + " \'" + str(value) + "\'", shell=True)
        p.wait()
            
    def setAudioInputSource(self):
        logMsg = "Audio input: {}".format(self.audioInput)
        self.logger.info(logMsg)
        syslog.syslog(logMsg)
        self.setMixerCParam("ADC Gain Capture Volume", 3)
        self.setMixerCParam('Mic1 Boost Volume', 3)
        self.setMixerCParam('Mic2 Capture Switch', 'off,off')
        self.setMixerCParam('Mic2 Boost Volume', 0)
        self.setMixerCParam("Line In Capture Switch", "off,on")
        self.setMixerCParam("Mic1 Capture Switch", "on,off")
        # self.setMixerParam("Mic1 Boost", str(str(self.micInGain) + "dB"))

    def initIOs(self):
        self.io1 = InOut(PIN3)
        self.io2 = InOut(PIN4)
        self.io3 = InOut(PIN2)
        self.io4 = InOut(PIN1)

        self.io5 = InOut(PIN5)
        self.io6 = InOut(PIN6)
        self.io7 = InOut(PIN7)
        self.io8 = InOut(PIN8)

        self.io9 = InOut(PIN9)      # Amp control
        self.io12 = InOut(PIN12)    # Amp status

        self.io9.setRelay(True)

    def run(self):
        ret_val = 0
        self.setLed2(0)

        try:
            self.reloadSettings()

            uci = Uci()
            fwUrl = uciGet(uci, "epic", "main", "fw_update_url")
            fwAutoUpdate = uciGet(uci, "epic", "main", "fw_update_auto")
            if fwUrl is not None and fwAutoUpdate == "true":
                self.fwUpdater = FwUpdater(fwUrl, "/mnt/data/state/epic-sip/fw_last_modified.conf")
                self.fwUpdater.setIniWindowTime(900)
                self.fwUpdater.setUpdateBaseTime(2, 0)
                self.fwUpdater.setUpdateWindowDuration(120)
            else:
                self.fwUpdater = None

            self.initIOs()

            self.redLed = leds.Led("barix:led1:red")
            self.greenLed = leds.Led("barix:led1:green")
            self.greenLed.set(True)
            self.redLed.set(False)
            syslog.syslog("RED LED --> OFF")
            syslog.syslog("GREEN LED --> ON")

            self.mainVolume = int(uci.get("epic", "main", "volume"))
            if self.mainVolume < 0:
                self.mainVolume = 0
            elif self.mainVolume > 100:
                self.mainVolume = 100

            self.mainVolumeCtl = alsaaudio.Mixer("BARIX_MAIN")
            syslog.syslog("MAIN VOLUME --> {}%".format(self.mainVolume))
            self.mainVolumeCtl.setvolume(self.mainVolume)
            
            self.bringUpMixers()
            
            self.httpApi = HttpApi("127.0.0.1", EpicSip.HTTP_API_PORT)
            self.httpApi.setXDHandler(self.xdHandler)
            self.httpApi.setAbclHandler(self.executeAbcl)
            self.httpApi.setStatusHandler(self.getAppStatus)
            self.httpApi.setReloadSettingsHandler(self.reloadSettings)

            self.httpApi.launch()

            tcpApiPort = int(uciGet(uci, "epic", "api", "abcl_tcp_port"))
            if tcpApiPort != 0:
                self.tcpApi = TcpApi("0.0.0.0", tcpApiPort)
                self.tcpApi.setAbclHandler(self.executeAbcl)
                self.tcpApi.launch()
            else:
                self.tcpApi = None

            self.playbackManager = PlaybackManager()

            bgmUrl = uciGet(uci, "epic", "bgm", "url")
            bgmBufSz = int(uci.get("epic", "bgm", "buffer_size"))
            bgmVol = int(uci.get("epic", "bgm", "volume"))
            if bgmUrl is not None and len(bgmUrl) > 0:
                urlComps = urlparse(bgmUrl)
                if urlComps.scheme == "rtp":
                    bgmParams = dict()
                    bgmParams["url"] = bgmUrl
                    bgmParams["playback_mode"] = "full_buffer"
                    bgmParams["input_buffer_size"] = 0
                    bgmParams["output_buffer_size"] = bgmBufSz
                    bgmSrc = RtpAudioSource(1, "bgm", bgmVol, bgmParams, "chan_1", 48000)
                    logMsg = "BGM SOURCE: {} buff={}ms vol={}%".format(bgmUrl, bgmBufSz, bgmVol)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.playbackManager.setBgmSource(bgmSrc)
                else:
                    self.logger.error("BGM url is not supported: {}".format(bgmUrl))

            hpnUrl = uciGet(uci, "epic", "hpn", "url")

            if hpnUrl is not None and len(hpnUrl) > 0:
                hpnBufSz = int(uci.get("epic", "hpn", "buffer_size"))
                hpnVol = int(uci.get("epic", "hpn", "volume"))
                if uci.get("epic", "hpn", "high_prio").lower() == "true":
                    self.hpnHighPrio = True
                else:
                    self.hpnHighPrio = False

                urlComps = urlparse(hpnUrl)
                if urlComps.scheme == "rtp":
                    hpnParams = dict()
                    hpnParams["url"] = hpnUrl
                    hpnParams["playback_mode"] = "low_latency"
                    hpnParams["input_buffer_size"] = 0
                    hpnParams["output_buffer_size"] = hpnBufSz
                    hpnSrc = RtpAudioSource(2, "hpn", hpnVol, hpnParams, "chan_2", 48000)
                    self.playbackManager.setHpnSource(hpnSrc)
                    logMsg = "HPN SOURCE: {} buff={}ms vol={}%".format(hpnUrl, hpnBufSz, hpnVol)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.playbackManager.setHpnHighPrio(self.hpnHighPrio)
                else:
                    self.logger.error("HPN url is not supported: {}".format(hpnUrl))

            self.autodial1 = uciGet(uci, "epic", "sip", "autodial1")
            self.autodial2 = uciGet(uci, "epic", "sip", "autodial2")
            self.autodial3 = uciGet(uci, "epic", "sip", "autodial3")
            self.autodial4 = uciGet(uci, "epic", "sip", "autodial4")

            sipEnable = uciGet(uci, "epic", "sip", "enabled")
            self.audioInput = uciGet(uci, "epic", "sip", "input")
            self.micInGain = uciGet(uci, "epic", "sip", "micin_gain")
            self.lineInGain = uciGet(uci, "epic", "sip", "linein_gain")
            sipVol = int(uci.get("epic", "sip", "volume"))
            sipDomain = uciGet(uci, "epic", "sip", "domain")
            self.sipUserName = uciGet(uci, "epic", "sip", "username")
            sipPassword = uciGet(uci, "epic", "sip", "password")
            autoAnswer = uciGet(uci, "epic", "sip", "auto_answer")
            answerTime = uciGet(uci, "epic", "sip", "answer_time")
            callTimeout = uciGet(uci, "epic", "sip", "call_timeout")
            streamTimeout = uciGet(uci, "epic", "sip", "stream_timeout")
            sipTransport = uciGet(uci, "epic", "sip", "transport")
            beepOnAnswer = uciGet(uci, "epic", "sip", "beep_on_answer")
            halfDuplex = uciGet(uci, "epic", "sip", "half_duplex")
            aecEnabled = uciGet(uci, "epic", "sip", "aec")
            inputGain = 0.01 * int(uciGet(uci, "epic", "sip", "input_gain"))
            hdxLevel = float(uciGet(uci, "epic", "sip", "hdx_level"))
            hdxTimeout = 0.001 * float(uciGet(uci, "epic", "sip", "hdx_timeout"))

            self.setAudioInputSource()

            self.sipSrc = None
            if sipEnable == "true":
                if sipDomain is not None and self.sipUserName is not None and sipPassword is not None:
                    sipParams = dict()
                    sipParams["sip_mode"] = "server"
                    sipParams["domain"] = sipDomain
                    sipParams["user_name"] = self.sipUserName
                    sipParams["password"] = sipPassword
                    sipParams["transport_and_encryption"] = sipTransport
                    sipParams["auto_answer"] = autoAnswer
                    sipParams["answer_time"] = answerTime
                    sipParams["call_timeout"] = callTimeout
                    sipParams["stream_timeout"] = streamTimeout
                    sipParams["beep_on_answer"] = beepOnAnswer
                    sipParams["half_duplex"] = halfDuplex
                    sipParams["aec"] = aecEnabled
                    sipParams["input_gain"] = inputGain
                    sipParams["hdx_level"] = hdxLevel
                    sipParams["hdx_timeout"] = hdxTimeout
                    self.logger.info("loading SIP with '{}'".format(sipParams))
                    self.sipSrc = SipClient(3, "sip", sipVol, sipParams, "chan_3", "default")
                    self.playbackManager.setSipSource(self.sipSrc)
                    syslog.syslog("Loaded SIP: domain='{}', user='{}'".format(sipDomain, self.sipUserName))
                else:
                    self.logger.error("SIP is enabled but some parameters are missing!")

            self.setAudioInputSource()

            self.serialGw = SerialGw(SerialGw.SERVER)
            serialGwMode = uciGet(uci, "epic", "serialgw", "mode")
            serialPeerPort = int(uciGet(uci, "epic", "serialgw", "port"))
            serialPeerHost = uciGet(uci, "epic", "serialgw", "host")

            if serialGwMode == "server":
                self.serialGw = SerialGw(SerialGw.SERVER)
            elif serialGwMode == "client":
                self.serialGw = SerialGw(SerialGw.CLIENT)
            elif serialGwMode == "disabled":
                self.serialGw = SerialGw(SerialGw.NOGW)
            else:
                self.logger.error("unknown serialgw mode: {}".format(serialGwMode))

            self.serialGw.setPeer(serialPeerPort, serialPeerHost)
            if self.serialConfig[0] is not None:
                baud, parity, databits, stopbits = self.serialConfig[0].split(":")
                self.serialGw.setSerialParams(0, int(baud), int(databits), parity, int(stopbits))
            if self.serialConfig[1] is not None:
                baud, parity, databits, stopbits = self.serialConfig[1].split(":")
                self.serialGw.setSerialParams(1, int(baud), int(databits), parity, int(stopbits))

            self.fwRP = uciGet(uci, "ae_ms", "serial", "egress_list_1")
            if self.fwRP is not None:
                self.fwRP = self.fwRP.split(",")
                self.serialGw.FW_PORT_RP = []
                for t in self.fwRP:
                    self.serialGw.FW_PORT_RP.append(int(t))
            self.fwXD = uciGet(uci, "ae_ms", "serial", "egress_list_5")
            if self.fwXD is not None:
                self.fwXD = self.fwXD.split(",")
                self.serialGw.FW_PORT_XD = []
                for t in self.fwXD:
                    self.serialGw.FW_PORT_XD.append(int(t))
            self.fwTCP = uciGet(uci, "ae_ms", "serial", "egress_list_6")
            if self.fwTCP is not None:
                self.fwTCP = self.fwTCP.split(",")
                self.serialGw.FW_PORT_TCP = []
                for t in self.fwTCP:
                    self.serialGw.FW_PORT_TCP.append(int(t))

            self.serialGw.device = self
            self.serialGw.start()
            print("Serial gateway started.")
            logMsg = "Loaded serial to TCP gateway: mode='{}'".format(serialGwMode)
            self.logger.info(logMsg)
            syslog.syslog(logMsg)

            self.playingNotification = False
            self.sendNotificationRelay1 = False
            self.sendNotificationRelay2 = False
            self.sendNotificationRelay3 = False
            self.sendNotificationRelay4 = False
            if uciGet(uci, "epic", "io", "notify_relay1") == "true":
                self.sendNotificationRelay1 = True
            if uciGet(uci, "epic", "io", "notify_relay2") == "true":
                self.sendNotificationRelay2 = True
            if uciGet(uci, "epic", "io", "notify_relay3") == "true":
                self.sendNotificationRelay3 = True
            if uciGet(uci, "epic", "io", "notify_relay4") == "true":
                self.sendNotificationRelay4 = True

            logMsg = "Initialization is complete! Entering main Loop..."
            self.logger.info(logMsg)
            syslog.syslog(logMsg)
            self.runMainLoop()
        except:
            self.logger.error("main loop finished with exception")
            traceback.print_exc()
            ret_val = 1

        self.logger.info("shutting down device...")
        if self.serialGw is not None:
            print("shutting down serial gw...")
            self.logger.info("shutting down serial gw...")
            self.serialGw.shutdown()
        if self.playbackManager:
            print("shutting down playback manager...")
            self.logger.info("shutting down playback manager...")
            self.playbackManager.shutdown()
        if self.httpApi:
            print("shutting down http API...")
            self.logger.info("shutting down http API...")
            self.httpApi.stop()
        if self.tcpApi:
            print("shutting down TCP api...")
            self.logger.info("shutting down TCP api...")
            self.tcpApi.shutdown()

        print("Exiting")
        sys.exit(ret_val)

    def buildDAE(self) -> str:
        with open("/sys/class/thermal/thermal_zone0/temp") as f:
            temp = int(f.readlines()[0].strip())
            temp = int(temp/1000)
        with open("/proc/uptime") as f:
            tmp=f.readlines()[0].strip()
            tmp=tmp.split(" ")
            uptime = int(float(tmp[0]))
            utDays = int(uptime/86400)
            utHours= int((uptime-utDays*86400)/3600)
            utMinutes=int((uptime-utDays*86400-utHours*3600)/60)
            uptime = "{:02d}{:02d}{:02d}".format(utDays, utHours, utMinutes)
        with open("/proc/meminfo") as f:
            tmp=f.readlines()[1].strip()
            tmp=tmp.split(":")[1].strip()
            tmp=tmp.split(" ")[0]
            mem = int(tmp)
            mem = int(mem/1024)
        with open("/sys/class/net/eth0/carrier_changes") as f:
            lnk_chng = int(f.readlines()[0].strip()) - 2
        message="$DAE:{}:{}:{}:{}\r\n"
        if temp>90:
            message="!!!!!!!!!!!!!!!$DAE:{}:{}:{}:{}!!!!!!!!!!!!!!!\r\n"
        elif temp>80:
            message=">>>>>>>>>>>>>>>$DAE:{}:{}:{}:{}<<<<<<<<<<<<<<<\r\n"
        elif temp>75:
            message="###############$DAE:{}:{}:{}:{}###############\r\n"
        message=message.format(temp, uptime, mem, lnk_chng)
        return message

    def readPots(self):
        self.ipPotValue=self.i2c.getInputVolume()
        if abs(self.ipPotPrevVal-self.ipPotValue) >= 2:
            self.ipPotPrevVal = self.ipPotValue
            self.setChannelVolume(self.mapInputChannel(5))

        self.opPotValue=self.i2c.getOutputVolume()
        if abs(self.opPotPrevVal-self.opPotValue) >= 2:
            self.opPotPrevVal = self.opPotValue
            self.setOutVolume(6)

    def runMainLoop(self):
        self.ledShowoff()

        loopCount = 0
        lastSipState = None
        btStatus = 0x0F

        prevAmpPower=0
        prevAmpStatus=0

        self.xdCount=0

        self.i2c.unmuteMS6266()
        self.reloadSettings()
        while not self.stopApplication:
            statusChanged = False

            if self.model not in LIMITTED_MODELS:
                if loopCount % 5 ==0:
                    self.readPots()

            # iterate fw update checks at ~5 seconds
            if self.fwUpdater is not None and loopCount % 50 == 0:
                self.fwUpdater.checkFwUpdates()

            if loopCount%10 == 0:
                ampPower = 1 if (self.io9.getMode()==0) else 0
                ampStatus= 1 if (self.io12.read()==0) else 0
                if (ampPower != prevAmpPower) or (ampStatus != prevAmpStatus):
                    notif="statechange,301,{},{}".format(ampPower, ampStatus)
                    self.tcpApi.broadcastStatusUpdate(notif)
                    prevAmpPower = ampPower
                    prevAmpStatus = ampStatus

            if self.xdActive:
                if loopCount%5 == 0:
                    if self.xdCount > 60:
                        self.xdActive=False
                        for led in [4, 3, 2, 1]:
                            self.padauk1.lightLED(led, 0, "white")

                    else:
                        self.xdSwap()
                        self.xdCount+=1

            evt = self.io3.checkEvents()
            if evt == InOut.PRESSED:
                btStatus &= 0x0E
                print("Bt1 pressed")
                if self.sipSrc is not None:
                    # if SIP is IDLE, dial self.autodial1 (if not None)
                    # answer if RINGING
                    # hangup if a call is active
                    currentSipState = self.sipSrc.reportSipCallState()
                    if currentSipState == SipCallState.IDLE and self.autodial1 is not None:
                        logMsg = "SIP AUTODIAL 1: calling {}".format(self.autodial1)
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.makeCall(self.autodial1)

                    if currentSipState == SipCallState.RINGING and self.sipSrc is not None:
                        logMsg = "SIP RINGING + INPUT1 --> ANSWER"
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.answerCall()

                    if currentSipState != SipCallState.IDLE and currentSipState != SipCallState.RINGING and self.sipSrc is not None:
                        logMsg = "SIP is not IDLE and not receiving a call + INPUT1 --> HANGUP"
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.hangupCall()

                statusChanged = True
                self.updateLED(1)
            elif evt == InOut.RELEASED:
                btStatus |= 0x01
                print("Bt1 released")
                statusChanged = True
                self.updateLED(1)

            evt = self.io4.checkEvents()
            if evt == InOut.PRESSED:
                btStatus &= 0x0D
                print("Bt2 pressed")
                statusChanged = True
                # if SIP is IDLE, dial self.autodial2 (if not None)
                # NB: do not answer if ringing
                if self.sipSrc is not None:
                    currentSipState = self.sipSrc.reportSipCallState()
                    if currentSipState == SipCallState.IDLE and self.autodial2 is not None and self.sipSrc is not None:
                        logMsg = "SIP AUTODIAL 2: calling {}".format(self.autodial2)
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.makeCall(self.autodial2)
                self.updateLED(2)
            elif evt == InOut.RELEASED:
                btStatus |= 0x02
                print("Bt2 released")
                statusChanged = True
                self.updateLED(2)

            # Inputs 3 and 4 only exist on CZ300 board
            evt = self.io1.checkEvents()
            if evt == InOut.PRESSED:
                btStatus &= 0x0B
                statusChanged = True
                print("Bt3 pressed")
                if self.sipSrc is not None:
                    currentSipState = self.sipSrc.reportSipCallState()
                    if currentSipState == SipCallState.IDLE and self.autodial3 is not None and self.sipSrc is not None:
                        logMsg = "SIP AUTODIAL 3: calling {}".format(self.autodial3)
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.makeCall(self.autodial3)
                self.updateLED(3)
            elif evt == InOut.RELEASED:
                btStatus |= 0x04
                statusChanged = True
                print("Bt3 released")
                self.updateLED(3)

            evt = self.io2.checkEvents()
            if evt == InOut.PRESSED:
                btStatus &= 0x07
                statusChanged = True
                print("Bt4 pressed")
                if self.sipSrc is not None:
                    currentSipState = self.sipSrc.reportSipCallState()
                    if currentSipState == SipCallState.IDLE and self.autodial4 is not None and self.sipSrc is not None:
                        logMsg = "SIP AUTODIAL 4: calling {}".format(self.autodial4)
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.makeCall(self.autodial4)
                self.updateLED(4)
            elif evt == InOut.RELEASED:
                btStatus |= 0x08
                statusChanged = True
                print("Bt4 released")
                self.updateLED(4)

            if statusChanged:
                self._buttonStatus = btStatus

            notificationState = self.playbackManager.hpnIsActive() or self.playbackManager.sipIsActive() or self.playbackManager.bgmIsActive()

            if self.playingNotification != notificationState:

                # statusChanged   = True
                if notificationState:
                    chi = self.mapInputChannel(100)
                    if self.channelEnabled[chi]:
                        self.setLed2(1)
                        stateStr = "ON"
                        logMsg = "Notify DSP, RED LED --> ON (RED)"
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        print("HPN|SIP stream started.")
                        self.handleAeCommand("$SXI:100\r\n")
                else:
                    self.setLed2(0)
                    stateStr = "OFF"
                    logMsg = "Notify DSP, RED LED --> OFF (OFF)"
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    print("HPN&SIP stream ended.")
                    self.restoreChannelState()

                if self.sendNotificationRelay1:
                    logMsg = "RELAY1 --> {}".format(stateStr)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.setRelay3(notificationState)

                if self.sendNotificationRelay2:
                    logMsg = "RELAY2 --> {}".format(stateStr)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.setRelay4(notificationState)

                if self.sendNotificationRelay3:
                    logMsg = "RELAY3 --> {}".format(stateStr)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.setRelay1(notificationState)

                if self.sendNotificationRelay4:
                    logMsg = "RELAY4 --> {}".format(stateStr)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.setRelay2(notificationState)

                self.playingNotification = notificationState

            if self.sipSrc is not None:
                newSipState = self.sipSrc.reportStatus()
                sipCallState = newSipState["callState"]
                if lastSipState is None or sipCallState != lastSipState["callState"]:
                    # statusChanged=True
                    stateStr = "????"
                    if sipCallState == SipCallState.IDLE:
                        stateStr = "IDLE"
                    elif sipCallState == SipCallState.CALLING:
                        stateStr = "CALLING dest='{}'".format(newSipState["currentCall"])
                    elif sipCallState == SipCallState.INCOMMING_CALL:
                        stateStr = "INCOMMING"
                    elif sipCallState == SipCallState.RINGING or sipCallState == SipCallState.CALLING_RINGING:
                        stateStr = "RINGING peer='{}'".format(newSipState["currentCall"])
                    elif sipCallState == SipCallState.CONNECTED:
                        stateStr = "CONNECTED peer='{}'".format(newSipState["currentCall"])
                    elif sipCallState == SipCallState.GETTING_NOTIF:
                        stateStr = "NOTIFICATION"
                    logMsg = "SIP Call State --> {}".format(stateStr)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)

                registered = newSipState["registered"]
                serverRespCode = newSipState["sipResponseCode"]
                if lastSipState is None or registered != lastSipState["registered"] or serverRespCode != lastSipState[
                    "sipResponseCode"]:
                    logMsg = "SIP Registration: {}, {}".format(registered.upper(), serverRespCode)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                lastSipState = newSipState

            if statusChanged:
                change = self.getInputChange()
                if change:
                    self.logger.info("broadcasting status update: {}".format(change))
                    self.tcpApi.broadcastStatusUpdate(change)

            loopCount += 1
            time.sleep(0.1)

        print("Exiting main loop")

    def handleSpecialCase(self, line: str):
        tokens = line.strip().split(":")
        cmd = tokens[0]
        ch = 1
        chi = 5
        if cmd == "$IVI" or cmd=="$CVU":
            if self.channelVolume[chi] < 16:
                self.channelVolume[chi] += 5
                if self.channelVolume[chi] > 16:
                    self.channelVolume[chi] = 16
                self.logger.info("Setting channel {} volume to {}.".format(tokens[1], self.channelVolume[chi]))
                self.storeChannelConfig(ch, chi)
                self.sendNotification("$CVI:{}:{}\r\n".format(1, self.channelVolume[chi]))
            self.serialGw.data4skt += line + "\r\n"
        elif cmd == "$DVI" or cmd=="$CVD":
            if self.channelVolume[chi] > -79:
                self.channelVolume[chi] -= 5
                if self.channelVolume[chi] < -79:
                    self.channelVolume[chi] = -79
                self.logger.info("Setting channel {} volume to {}.".format(ch, self.channelVolume[chi]))
                self.storeChannelConfig(ch, chi)
                self.sendNotification("$CVI:{}:{}\r\n".format(1, self.channelVolume[chi]))
            self.serialGw.data4skt += line + "\r\n"

    def handleAeCommand(self, line: str):
        self.logger.info("AE command: '{}'".format(line.strip()))
        tokens = line.strip().split(":")
        cmd = tokens[0]
        if cmd == "$SVI":
            if len(tokens) != 3:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            if ch not in AUDIO_INPUTS :
                self.logger.error("Invalid input channel#.")
                self.serialGw.data4skt = "$$$"
                return
            chi = self.mapInputChannel(ch)

            vol = int(tokens[2])
            self.logger.info("Setting channel {} volume to {}.".format(ch, vol))
            self.channelVolume[chi] = vol
            self.storeChannelConfig(ch, chi)

            self.sendNotification("$CVI:{}:{}\r\n".format(ch, self.channelVolume[chi]))
            self.serialGw.data4skt += line + "\r\n"
        elif cmd == "$IVI":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            if tokens[1]=="9":
                self.handleSpecialCase(line)
                return

            ch = int(tokens[1])
            if ch not in AUDIO_INPUTS :
                self.logger.error("Invalid input channel#")
                self.serialGw.data4skt = "$$$"
                return
            chi = self.mapInputChannel(ch)

            if self.channelVolume[chi] < 16:
                self.channelVolume[chi] += 5
                if self.channelVolume[chi] > 16:
                    self.channelVolume[chi] = 16
                self.logger.info("Setting channel {} volume to {}.".format(tokens[1], self.channelVolume[chi]))
                self.storeChannelConfig(ch, chi)
                self.sendNotification("$CVI:{}:{}\r\n".format(tokens[1], self.channelVolume[chi]))
            self.serialGw.data4skt += line + "\r\n"
        elif cmd == "$DVI":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            if tokens[1]=="9":
                self.handleSpecialCase(line)
                return

            ch = int(tokens[1])
            if ch not in AUDIO_INPUTS :
                self.logger.error("Invalid input channel#")
                self.serialGw.data4skt = "$$$"
                return
            chi = self.mapInputChannel(ch)

            if self.channelVolume[chi] > -79:
                self.channelVolume[chi] -= 5
                if self.channelVolume[chi] < -79:
                    self.channelVolume[chi] = -79
                self.logger.info("Setting channel {} volume to {}.".format(ch, self.channelVolume[chi]))
                self.storeChannelConfig(ch, chi)
                self.sendNotification("$CVI:{}:{}\r\n".format(ch, self.channelVolume[chi]))
            self.serialGw.data4skt += line + "\r\n"
        elif cmd == "$SEI":
            if len(tokens) != 3:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            if ch not in AUDIO_INPUTS :
                self.logger.error("Invalid input channel#.")
                self.serialGw.data4skt = "$$$"
                return
            chi = self.mapInputChannel(ch)

            en = True if tokens[2] == "1" else False
            self.logger.info("Set enable input channel {} -> {}.".format(ch, en))
            self.channelEnabled[chi] = en
            self.preSXIstate[chi] = en
            self.storeChannelConfig(ch, chi)

            self.serialGw.data4skt += line + "\r\n"
            self.sendNotification("$CEI:{}:{}\r\n".format(tokens[1], tokens[2]))
        elif cmd == "$QVI":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            if ch not in AUDIO_INPUTS :
                self.logger.error("Invalid input channel#.")
                self.serialGw.data4skt = "$$$"
                return
            chi = self.mapInputChannel(ch)

            vol = self.channelVolume[chi]
            self.serialGw.data4skt += "$QVI:{}:{}\r\n".format(ch, vol)
        elif cmd == "$QEI":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            if ch not in AUDIO_INPUTS :
                self.logger.error("Invalid input channel#.")
                self.serialGw.data4skt = "$$$"
                return
            chi = self.mapInputChannel(ch)

            en = 1 if self.channelEnabled[chi] else 0
            self.serialGw.data4skt += "$QEI:{}:{}\r\n".format(ch, en)
        elif cmd == "$SXI":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            if ch not in AUDIO_INPUTS :
                self.logger.error("Invalid input channel#.")
                self.serialGw.data4skt = "$$$"
                return
            chi = self.mapInputChannel(ch)

            self.logger.info("Setting channel {} as exclusive.".format(ch))
            for ci in (1, 2, 3, 4):
                self.preSXIstate[ci] = self.channelEnabled[ci]
                if ci == chi:
                    self.channelEnabled[ci] = True
                else:
                    self.channelEnabled[ci] = False
                self.setChannelVolume(ci)
            self.serialGw.data4skt += line + "\r\n"  # Only respond to the sender if it's coming from the gw
        elif cmd == "$SVO":
            if len(tokens) != 3:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            chi = self.mapOutputChannel(ch)
            if chi > 6 or chi < 5:
                self.logger.error("Invalid output channel#.")
                self.serialGw.data4skt = "$$$"
                return

            vol = int(tokens[2])
            self.logger.info("Setting channel {} volume to {}.".format(ch, vol))
            self.channelVolume[chi] = vol
            self.storeChannelConfig(ch, chi)
            self.sendNotification("$CVO:{}:{}\r\n".format(ch, vol))
            self.serialGw.data4skt += line + "\r\n"
        elif cmd == "$IVO":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            chi = self.mapOutputChannel(ch)
            if chi > 6 or chi < 5:  # Channels 5 and 6 are the output channels
                self.logger.error("Invalid output channel#")
                self.serialGw.data4skt = "$$$"
                return

            if self.channelVolume[chi] < 16:
                self.channelVolume[chi] += 5
                if self.channelVolume[chi] > 16:
                    self.channelVolume[chi] = 16
                self.logger.info("Setting output channel {} volume to {}.".format(ch, self.channelVolume[chi]))
                self.storeChannelConfig(ch, chi)
                self.sendNotification("$CVO:{}:{}\r\n".format(ch, self.channelVolume[chi]))
            self.serialGw.data4skt += line + "\r\n"
        elif cmd == "$DVO":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            chi = self.mapOutputChannel(ch)
            if chi > 6 or chi < 5:  # Channels 5 and 6 are the output channels
                self.logger.error("Invalid output channel#")
                self.serialGw.data4skt = "$$$"
                return

            if self.channelVolume[chi] > -79:
                self.channelVolume[chi] -= 5
                if self.channelVolume[chi] < -79:
                    self.channelVolume[chi] = -79
                self.logger.info("Setting output channel {} volume to {}.".format(ch, self.channelVolume[chi]))
                self.storeChannelConfig(ch, chi)
                self.sendNotification("$CVO:{}:{}\r\n".format(ch, self.channelVolume[chi]))
            self.serialGw.data4skt += line + "\r\n"
        elif cmd == "$SEO":
            if len(tokens) != 3:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            chi = self.mapOutputChannel(ch)
            if chi > 6 or chi < 5:  # Channels 5 and 6 are the output channels
                self.logger.error("Invalid output channel#.")
                self.serialGw.data4skt = "$$$"
                return

            en = True if tokens[2] == "1" else False
            self.logger.info("Set enable output channel {} -> {}.".format(ch, en))
            self.channelEnabled[chi] = en
            self.storeChannelConfig(ch, chi)
            self.serialGw.data4skt += line + "\r\n"
            self.sendNotification("$CEO:{}:{}\r\n".format(ch, tokens[2]))
        elif cmd == "$QVO":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            chi = self.mapOutputChannel(ch)
            if chi > 6 or chi < 5:  # Channels 5 and 6 are the output channels
                self.logger.error("Invalid output channel#.")
                self.serialGw.data4skt = "$$$"
                return

            vol = self.channelVolume[chi]
            self.serialGw.data4skt += "$QVO:{}:{}\r\n".format(ch, vol)
        elif cmd == "$QEO":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            ch = int(tokens[1])
            chi = self.mapOutputChannel(ch)
            if chi > 6 or chi < 5:  # Channels 5 and 6 are the output channels
                self.logger.error("Invalid output channel#.")
                self.serialGw.data4skt = "$$$"
                return

            en = 1 if self.channelEnabled[chi] else 0
            self.serialGw.data4skt += "$QEO:{}:{}\r\n".format(ch, en)
        elif cmd=="$CVU":
            self.handleSpecialCase(line)
        elif cmd=="$CVD":
            self.handleSpecialCase(line)
        elif cmd == "$SPB":
            if len(tokens) != 3:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            p = int(tokens[1])
            b = int(tokens[2])
            if p not in [1, 5]:
                self.logger.error("Invalid serial port#.")
                self.serialGw.data4skt = "$$$"
                return
            # self.serialGw.setSerialParams(p, b)
            o = "config_{}".format(p)
            v = "{}:N:8:1".format(b)
            uciSet(Uci(), "ae_ms", "serial", o, v)
            self.serialGw.data4skt += line + "\r\n"
        elif cmd == "$QPB":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            if tokens[1] not in ['1', '5']:
                self.logger.error("Invalid serial port#.")
                self.serialGw.data4skt = "$$$"
                return
            p = int(tokens[1])
            p=[0, 0, 0, 0, 0, 1][p]     # Map pin1 to pin0 and pin5 to pin1
            b = 0
            if self.serialConfig[p] is not None:
                b, _, _, _ = self.serialConfig[p].split(":")
            self.serialGw.data4skt += "$QPB:{}:{}\r\n".format(tokens[1], b)
        elif cmd == "$QCC":
            if len(tokens) != 1:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            chi=self.mapInputChannel(4)
            reply = "$SEI:4:{}\r\n".format(1 if self.channelEnabled[chi] else 0)
            reply += "$SVI:4:{}\r\n".format(self.channelVolume[chi])
            chi=self.mapInputChannel(100)
            reply += "$SEI:100:{}\r\n".format(1 if self.channelEnabled[chi] else 0)
            reply += "$SVI:100:{}\r\n".format(self.channelVolume[chi])
            if self.model not in LIMITTED_MODELS:
                chi=self.mapInputChannel(5)
                reply += "$SEI:5:{}\r\n".format(1 if self.channelEnabled[chi] else 0)
                reply += "$SVI:5:{}\r\n".format(self.channelVolume[chi])
            chi=self.mapInputChannel(1)
            reply += "$SEI:1:{}\r\n".format(1 if self.channelEnabled[chi] else 0)
            reply += "$SVI:1:{}\r\n".format(self.channelVolume[chi])
            reply += "$SEO:1:{}\r\n".format(1 if self.channelEnabled[5] else 0)         # Mixer channel 5 maps to the output channel 1
            reply += "$SVO:1:{}\r\n".format(self.channelVolume[5])
            if self.model not in LIMITTED_MODELS:                                       # CZ-300
                reply += "$SEO:4:{}\r\n".format(1 if self.channelEnabled[6] else 0)     # Mixer channel 6 maps to the output channel 2
                reply += "$SVO:4:{}\r\n".format(self.channelVolume[6])
            else:                                                                       # CZ-301
                reply += "$SEO:0:{}\r\n".format(1 if self.channelEnabled[6] else 0)     # Mixer channel 6 maps to the output channel
                reply += "$SVO:0:{}\r\n".format(self.channelVolume[6])
            fwReply = ""
            if len(self.serialGw.FW_PORT_RP) > 0:
                for f in self.serialGw.FW_PORT_RP:
                    if len(fwReply) > 0:
                        fwReply += ","
                    fwReply += "{}".format(f)
            else:
                fwReply = "0"
            reply += "$SPF:1:{}\r\n".format(fwReply)
            fwReply = ""
            if len(self.serialGw.FW_PORT_XD) > 0:
                for f in self.serialGw.FW_PORT_XD:
                    if len(fwReply) > 0:
                        fwReply += ","
                    fwReply += "{}".format(f)
            else:
                fwReply = "0"
            reply += "$SPF:5:{}\r\n".format(fwReply)
            fwReply = ""
            if len(self.serialGw.FW_PORT_TCP) > 0:
                for f in self.serialGw.FW_PORT_TCP:
                    if len(fwReply) > 0:
                        fwReply += ","
                    fwReply += "{}".format(f)
            else:
                fwReply = "0"
            reply += "$SPF:6:{}\r\n".format(fwReply)
            if self.serialConfig[0] is not None:
                baud, _, _, _ = self.serialConfig[0].split(":")
                reply += "$SPB:1:{}\r\n".format(baud)
            if self.serialConfig[1] is not None:
                baud, _, _, _ = self.serialConfig[1].split(":")
                reply += "$SPB:2:{}\r\n".format(baud)
            self.serialGw.data4skt += reply
        elif cmd == "$SPF":
            if len(tokens) != 3:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            if not len(tokens) == 3:
                self.logger.error("Invalid command '{}'.".format(line))
                return
            src = int(tokens[1])
            dst = tokens[2].split(",")
            if src == 1:
                self.serialGw.FW_PORT_RP = []
                d_list = ""
                for d in dst:
                    if len(d_list) > 0:
                        d_list += ","
                    d_list += d
                    d = int(d.strip())
                    if d > 0:
                        self.serialGw.FW_PORT_RP.append(d)
                self.logger.info("Forward serial port RP to {}.".format(d_list))
                uciSet(Uci(), "ae_ms", "serial", "egress_list_1", d_list)
            elif src == 5:
                self.serialGw.FW_PORT_XD = []
                d_list = ""
                for d in dst:
                    if len(d_list) > 0:
                        d_list += ","
                    d_list += d
                    d = int(d.strip())
                    if d > 0:
                        self.serialGw.FW_PORT_XD.append(d)
                self.logger.info("Forward serial port XD to {}.".format(d_list))
                uciSet(Uci(), "ae_ms", "serial", "egress_list_5", d_list)
            elif src == 6:
                self.serialGw.FW_PORT_TCP = []
                d_list = ""
                for d in dst:
                    if len(d_list) > 0:
                        d_list += ","
                    d_list += d
                    d = int(d.strip())
                    if d > 0:
                        self.serialGw.FW_PORT_TCP.append(d)
                self.logger.info("Forward serial port TCP to {}.".format(d_list))
                uciSet(Uci(), "ae_ms", "serial", "egress_list_6", d_list)
            else:
                self.logger.error("Invalid serial port#.")
                return
            self.serialGw.data4skt += line + "\r\n"
        elif cmd=="$QPF":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            src=int(tokens[1])
            fwReply=""
            if src==1:
                for f in self.serialGw.FW_PORT_RP:
                    if len(fwReply) > 0:
                        fwReply += ","
                    fwReply += "{}".format(f)
                fwReply="$QPF:1:{}\r\n".format(fwReply)
            elif src==5:
                for f in self.serialGw.FW_PORT_XD:
                    if len(fwReply) > 0:
                        fwReply += ","
                    fwReply += "{}".format(f)
                fwReply = "$QPF:5:{}\r\n".format(fwReply)
            elif src==6:
                for f in self.serialGw.FW_PORT_TCP:
                    if len(fwReply) > 0:
                        fwReply += ","
                    fwReply += "{}".format(f)
                fwReply = "$QPF:6:{}\r\n".format(fwReply)
            if len(fwReply) > 0:
                self.serialGw.data4skt += fwReply
        elif cmd=="$SSP":
            if len(tokens) != 3:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return
            
            data = bytes.fromhex(tokens[2])
            src = self.serialGw.srcChannel
            self.logger.warning("handling '{}' from port {} to port {}".format(data.decode('utf-8').strip(), ['1', '5', '6'][src], tokens[1]))
            self.serialGw.srcChannel = ['1', '5', '6'].index(tokens[1])
            self.serialGw.data4skt += data.decode('utf-8').strip()+"\r\n"
        elif cmd=="$SSO":
            if len(tokens) != 2:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            st=False if tokens[1]=="1" else True
            self.io9.setRelay(st)
            self.serialGw.data4skt += line + "\r\n"
        elif cmd=="$QSO":
            if len(tokens) != 1:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            st=1 if self.io9.getMode()==0 else 0
            self.serialGw.data4skt += "$QSO:{}\r\n".format(st)
        elif cmd=="$QPW":
            if len(tokens) != 1:
                self.logger.error("Malformed command.")
                self.serialGw.data4skt = "$$$"
                return

            st=self.io12.read()
            st="$QPW:{}\r\n".format(st)
            self.serialGw.data4skt += st
        else:
            if line[:1] == "$":
                if self.serialGw.srcChannel==2:
                    self.serialGw.data4XD += line + "\r\n"
                    self.serialGw.srcChannel = -1
                else:
                    self.serialGw.data4skt += line + "\r\n"
                    self.serialGw.srcChannel = 2

            self.logger.error("Unknown AE command {}. Command forwarded.".format(line))

    def sendNotification(self, msg):
        self.serialGw.notification = msg

    def executeAbcl(self, line: str) -> str:
        if len(line) == 0:
            return ""

        self.logger.info("ABCL command: cmd='{}'".format(line))
        cmd0 = line[0]

#        print("Abcl: "+line)
        if cmd0=="S":
            self.logger.info("Toggling amp status (STDBY/ON)")
            print("Toggling amp status (STDBY/ON)")
            if self.io9.getMode()==0:
                self.io9.setRelay(True)
            else:
                self.io9.setRelay(False)
        elif cmd0 == "T":
            # all other arguments are ignored. might validate empty params...
            reply = dict()
            reply["m"] = self.model
            reply["s"] = "IPAM-400"
            reply["v"] = self.sysVersion
            reply["a"] = getMACaddr()
            return json.dumps(reply, indent=4)
        elif cmd0 == "D":
            (cmd, arg) = decodeDCommand(line)
            sipState = self.sipSrc.reportSipCallState()
            if cmd == "D":
                # dial SIP <arg> if SIP is IDLE
                if sipState == SipCallState.IDLE and self.sipSrc is not None:
                    logMsg = "ABCL Dial (D) --> dial '{}'".format(arg)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.sipSrc.makeCall(arg)
            elif cmd == "DS":
                # Hangup any currently active SIP call
                if self.sipSrc is not None:
                    logMsg = "ABCL Hangup (DS) --> hangup SIP call (if any)"
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.sipSrc.hangupCall()
            elif cmd == "DA":
                # answer the incoming SIP call
                if self.sipSrc is not None:
                    logMsg = "ABCL Answer (DA) --> Answer SIP call (if any)"
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.sipSrc.answerCall()
            elif cmd == "DH":
                # dial SIP <arg>
                # if a call is active, terminate it and dial the new one
                if self.sipSrc is not None:
                    if sipState != SipCallState.IDLE:
                        logMsg = "ABCL Dial-Prio (DH) --> hangup current call"
                        self.logger.info(logMsg)
                        syslog.syslog(logMsg)
                        self.sipSrc.hangupCall()
                    logMsg = "ABCL Dial-Prio (DH) --> dial '{}'".format(arg)
                    self.logger.info(logMsg)
                    syslog.syslog(logMsg)
                    self.sipSrc.makeCall(arg)
            else:
                self.logger.warning("unrecognized ABCL D*** command: '{}'".format(line))
        elif line.find("VS=") == 0:
            if len(line) < 4:
                self.logger.warning("malformed VS command: '{}'".format(line))
            else:
                vol = int(line[3:])
                if vol < 0 or vol > 20:
                    self.logger.warning("bad volume in VS command: '{}' (expected=0..20)".format(vol))
                    return ""
                self.mainVolume = vol * 5
                self.mainVolumeCtl.setvolume(self.mainVolume)
                logMsg = "ABCL Set Volume (VS) --> volume={} / {}%".format(vol, self.mainVolume)
                self.logger.info(logMsg)
                syslog.syslog(logMsg)
                uci = Uci()
                uci.set("epic", "main", "volume", str(self.mainVolume))
                uci.commit("epic")
        elif line == "VQ":
            return str(self.mainVolume // 5)
        elif line == "Q":
            msg = self.getDeviceStatus()
            msg += "\r\n"
            msg += self.buildDAE()
            return msg
        elif line.find("R1=") == 0 or line.find("R2=") == 0 or line.find("R3=") == 0 or line.find("R4=") == 0:
            if len(line) < 4:
                self.logger.warning("malformed R1/R2/R3/R4 command: '{}'".format(line))
                return ""

            relayNum = int(line[1])
            intVal = int(line[3])
            relayVal = True if intVal == 1 else False

            if relayNum == 1:
                self.setRelay3(relayVal)
                time.sleep(0.005)
                self.setRelay3(relayVal)
            elif relayNum == 2:
                self.setRelay4(relayVal)
                time.sleep(0.005)
                self.setRelay4(relayVal)
            elif relayNum == 3:
                self.setRelay1(relayVal)
                time.sleep(0.005)
                self.setRelay1(relayVal)
            else:
                self.setRelay2(relayVal)
                time.sleep(0.005)
                self.setRelay2(relayVal)
        elif cmd0 == "A":
            if line == "AR":  # Reload settings
                pass
            elif line == "AB":  # Reboot
                pass
        else:
            self.logger.warning("unrecognized ABCL command: '{}'".format(line))

        return ""

    xdStatus=False
    def xdSwap(self):
        if self.xdStatus:
            for led in [4, 3, 2, 1]:
                self.padauk1.lightLED(led, 0, "white")
            self.xdStatus=False
        else:
            for led in [4, 3, 2, 1]:
                self.padauk1.lightLED(led, 4, "white")
            self.xdStatus=True

    def xdHandler(self):
        for led in [4, 3, 2, 1]:
            self.padauk1.lightLED(led, 4, "white")
        self.xdStatus=True
        self.xdActive=True

    def getInputChange(self) -> str:
        #        print('{} {}'.format(self._buttonPStatus, self._buttonStatus))
        dif = self._buttonStatus ^ self._buttonPStatus

        if dif == 0:
            return ""

        self._buttonPStatus = self._buttonStatus

        st = ~self._buttonStatus
        curr = 1
        forBC = ''

        for n in [1, 2, 3, 4]:
            if dif & curr:
                change = st & curr
                if change > 0:
                    change = 1
                if forBC:
                    forBC = forBC + '\r\n'
                forBC = forBC + 'statechange,20{},{}'.format(n, change)

            curr = curr << 1

        return forBC


    def getDeviceStatus(self) -> str:
        """
        * "SIP_ID": sip user id
        * STT:
            - S0 --> IDLE
            - S1 --> CALLING
            - S2 --> INCOMING CALL / RINGING
            - S3 --> CONNECTED
            - S4 --> GETTING NOTIFICATION??????
        """
        sipUid = "0"  # invalid SIP ID
        sipState = "S0"

        if self.sipSrc is not None:
            sipUid = self.sipUserName
            state = self.sipSrc.reportSipCallState()
            if state == SipCallState.IDLE:
                sipState = "S0"
            elif state == SipCallState.CALLING:
                sipState = "S1"
            elif state == SipCallState.INCOMMING_CALL or state == SipCallState.RINGING or state == SipCallState.CALLING_RINGING:
                sipState = "S2"
            elif state == SipCallState.CONNECTED:
                sipState = "S3"
            elif state == SipCallState.GETTING_NOTIF:
                sipState = "S4"

        ii = 0x0
        if self.io1.read() != 0:
            ii |= 0x1
        if self.io2.read() != 0:
            ii |= 0x2
        if self.io3.read() != 0:
            ii |= 0x4
        if self.io4.read() != 0:
            ii |= 0x8

        oo = 0
        if self.io5.getMode() == 1:
            oo |= 0x1
        if self.io6.getMode() == 1:
            oo |= 0x2
        if self.io7.getMode() == 1:
            oo |= 0x4
        if self.io8.getMode() == 1:
            oo |= 0x8

        return "{},{},{:02X},{:02X}".format(sipUid, sipState, ii, oo)

    def getAppStatus(self):
        audioStatus = "IDLE"

        sipStatus = "NOT CONFIGURED"
        sipRegStatus = None
        sipRegistrationCode = None
        sipPeer = None
        if self.sipSrc is not None:
            currSipStatus = self.sipSrc.reportStatus()
            sipRegStatus = str(currSipStatus["registered"]).upper()
            sipStatus = self.sipDict[currSipStatus["callState"]]
            sipRegistrationCode = currSipStatus["sipResponseCode"]
            sipPeer = currSipStatus["currentCall"]
            if self.playbackManager.sipIsActive():
                audioStatus = "SIP"

        hpnStatus = "NOT CONFIGURED"
        if self.playbackManager.hpnSource is not None:
            if self.playbackManager.hpnIsActive():
                hpnStatus = "ALIVE"
                audioStatus = "HPN"
            else:
                (state, alive) = self.playbackManager.hpnSource.getState()
                if alive:
                    hpnStatus = "ALIVE"
                else:
                    hpnStatus = "IDLE"

        bgmStatus = "NOT CONFIGURED"
        if self.playbackManager.bgmSource is not None:
            if self.playbackManager.bgmIsActive():
                bgmStatus = "ALIVE"
                audioStatus = "BGM"
            else:
                (state, alive) = self.playbackManager.bgmSource.getState()
                if alive:
                    bgmStatus = "ALIVE"
                else:
                    bgmStatus = "IDLE"

        ioStatus = dict()
        ioStatus["in1"] = "ON" if self.io3.read()==0 else "OFF"
        ioStatus["in2"] = "ON" if self.io4.read()==0 else "OFF"
        ioStatus["in3"] = "ON" if self.io1.read()==0 else "OFF"
        ioStatus["in4"] = "ON" if self.io2.read()==0 else "OFF"

        ioStatus["out1"] = "ON" if self.io5.getMode()==1 else "OFF"
        ioStatus["out2"] = "ON" if self.io6.getMode()==1 else "OFF"
        ioStatus["out3"] = "ON" if self.io7.getMode()==1 else "OFF"
        ioStatus["out4"] = "ON" if self.io8.getMode()==1 else "OFF"

        ampStatus = dict()
        ampStatus["power"] = 0 if self.io9.getMode()==1 else 1
        ampStatus["enabled"] = (self.io12.read()==0)
        # print(ampStatus)

        status = {
            "sipStatus": "{}".format(sipStatus),
            "sipRegStatus": "{}".format(sipRegStatus),
            "sipRegistrationCode": "{}".format(sipRegistrationCode),
            "sipPeer": "{}".format(sipPeer),
            "hpnStatus": "{}".format(hpnStatus),
            "bgmStatus": "{}".format(bgmStatus),
            "audioStatus": "{}".format(audioStatus),
            "ioStatus": ioStatus,
            "ampStatus": ampStatus
        }
        status = json.dumps(status).encode('utf-8')
        return status

if __name__ == "__main__":
    rotHandler = RotatingFileHandler("/var/log/epic.log", maxBytes=2 * 1024 * 1024, backupCount=3)
    logging.basicConfig(level=logging.DEBUG, handlers=[rotHandler],
                        format='%(asctime)s %(name)s %(levelname)s %(message)s')
    baco.initializeLogger("/var/log/baco.log", baco.LogPriority.INFO, 2000000, 5, False)
    found, _ = findLicense("core-image-ae-msx00")
    if not found:
        found, _ = findLicense("core-image-ae-cz300")
    if not found:
        found, _ = findLicense("barix-development-platform")

    if not found:
        l = logging.getLogger("main")
        l.error("required license is missing: 'core-image-ae-msx00|barix-development-platform'. Exit.")
        sys.exit(1)

    baco.initializeAudioLib()

    print("Checking device model")
    with open("MODEL", "r") as f:
        model = f.read()
    model = model.strip()

    if (model == 'CZ300'):
        isCZ300 = True
        print("CZ300 detected")

    app = EpicSip()
    app.run()
    print("ALL FINISHED")
