
import time
import syslog

from uci import *
import json
from threading import *
import os
import logging
from barix.media.AudioSource import *


class PlaybackManager:
    def __init__(self):
        self.lock           = Lock()
        self.workerCond     = Condition(lock=self.lock)
        self.stopThread     = False
        self.logger         = logging.getLogger('playback-mgr')
        self.bgmSource      = None
        self.hpnSource      = None
        self.hpnHighPrio    = False
        self.sipSource      = None
        self.activeSource   = None
        self.workerThread   = Thread(target=self.workerThreadFunction)
        self.workerThread.start()


    def shutdown(self):
        try:
            self.workerCond.acquire()
            self.stopThread     = True
            self.workerCond.notify_all()
        finally:
            self.workerCond.release()
        self.workerThread.join()

    def workerThreadFunction(self):
        self.logger.info("worker thread starting")
        while not self.stopThread:
            self.refreshSources()
            try:
                self.workerCond.acquire()
                self.workerCond.wait(0.5)
            finally:
                self.workerCond.release()

        if self.bgmSource is not None:
            self.logger.info("purging bgm source...")
            self.bgmSource.stopAudio()
            self.bgmSource  = None

        if self.hpnSource is not None:
            self.logger.info("purging HPN...")
            self.hpnSource.stopAudio()
            self.hpnSource  = None

        if self.sipSource is not None:
            self.logger.info("purging sip source...")
            self.sipSource.stopAudio()
            self.sipSource  = None

        self.logger.info("worker thread stopping")

    def sourceStateCallback(self,sourceId,sourceState,alive):
        try:
            self.workerCond.acquire()
            self.logger.debug("source event: id={}, state={}, alive={}".format(sourceId, sourceState, alive))
            self.workerCond.notify_all()
        finally:
            self.workerCond.release()


    def refreshSources(self):
        try:
            self.lock.acquire()
            self.innerRefreshSources()
        finally:
            self.lock.release()

    def innerRefreshSources(self):
        bgmAlive    = False
        bgmState    = None
        hpnAlive    = False
        hpnState    = None
        sipAlive    = False
        sipState    = None

        if self.bgmSource is not None:
            (bgmState, bgmAlive) = self.bgmSource.getState()
            if bgmState==SourceState.STOPPED:
                self.bgmSource.checkAudio()

        if self.hpnSource is not None:
            (hpnState, hpnAlive) = self.hpnSource.getState()
            if hpnState==SourceState.STOPPED:
                self.hpnSource.checkAudio()

        if self.sipSource is not None:
            (sipState, sipAlive) = self.sipSource.getState()
            if sipState==SourceState.STOPPED:
                self.sipSource.checkAudio()

        sourceToActivate    = None
        if bgmAlive:
            sourceToActivate    = self.bgmSource

        if hpnAlive:
            sourceToActivate    = self.hpnSource

        if sipAlive:
            sourceToActivate    = self.sipSource

        if hpnAlive and self.hpnHighPrio:
            sourceToActivate    = self.hpnSource

        if self.activeSource != sourceToActivate:
            oldSrc  = "<None>"
            if self.activeSource is not None:
                oldSrc  = self.activeSource.getName()

            newSrc  = "<None>"
            if sourceToActivate is not None:
                newSrc  = sourceToActivate.getName()

            logMsg  = "Switching Source: {} --> {}".format(oldSrc,newSrc)
            self.logger.info(logMsg)
            syslog.syslog(logMsg)
            if self.bgmSource is not None and sourceToActivate!=self.bgmSource:
                self.bgmSource.checkAudio()

            if self.hpnSource is not None and sourceToActivate!=self.hpnSource:
                self.hpnSource.checkAudio()
            
            if self.sipSource is not None and sourceToActivate!=self.sipSource:
                self.sipSource.checkAudio()

            if sourceToActivate is not None:
                sourceToActivate.playAudio()
            self.activeSource   = sourceToActivate



    def setBgmSource(self,src:AudioSource):
        """
        Sets the audio source for the background channel.
        src must be a subclass of AudioSource
        """
        try:
            self.lock.acquire()
            src.addSourceStateObserver(self.sourceStateCallback)
            self.bgmSource  = src
            self.workerCond.notify_all()
        finally:
            self.lock.release()

    def setHpnSource(self,src:AudioSource):
        try:
            self.lock.acquire()
            src.addSourceStateObserver(self.sourceStateCallback)
            self.hpnSource  = src
            self.workerCond.notify_all()
        finally:
            self.lock.release()

    def setHpnHighPrio(self,highPrio:bool):
        self.hpnHighPrio = highPrio

    def setSipSource(self,src:AudioSource):
        try:
            self.lock.acquire()
            src.addSourceStateObserver(self.sourceStateCallback)
            self.sipSource  = src
            self.workerCond.notify_all()
        finally:
            self.lock.release()
    
    def bgmIsActive(self) -> bool:
        """
        Checks if BGM is the active source.
        """
        try:
            self.lock.acquire()
            return self.bgmSource is not None and self.bgmSource==self.activeSource
        finally:
            self.lock.release()


    def sipIsActive(self) -> bool:
        try:
            self.lock.acquire()
            return self.sipSource is not None and self.sipSource==self.activeSource
        finally:
            self.lock.release()


    def hpnIsActive(self) -> bool:
        try:
            self.lock.acquire()
            return self.hpnSource is not None and self.hpnSource==self.activeSource
        finally:
            self.lock.release()
