import hmac
from hashlib import md5
from flask import (Blueprint, request)
import werkzeug.datastructures
# External API authentication
from flask_httpauth import HTTPDigestAuth
from .ae_utils import is_ae_uci_change_immediately
from .common import *
from .internal_funcs import *

DEVICE_MODEL = getDeviceModel()

bp = Blueprint('external_API', __name__)
users = ["admin"]

NONCE = None
OPAQUE = None

auth = HTTPDigestAuth(realm=REALM, use_ha1_pw=True)

@auth.get_password
def get_pw(username):
    if username in users:
        password = getWebUIPwd(username)
        return password
    return None

from random import SystemRandom


def getRandom():
    random = SystemRandom()
    return md5(str(random.random()).encode('utf-8')).hexdigest()


@auth.generate_nonce
def generate_nonce():
    global NONCE
    nonce = getRandom()
    NONCE = nonce
    return nonce


@auth.generate_opaque
def generate_opaque():
    global OPAQUE
    opaque = getRandom()
    OPAQUE = opaque
    return opaque


@auth.verify_nonce
def verify_nonce(nonce):
    if nonce is None or NONCE is None:
        return False
    return hmac.compare_digest(nonce, NONCE)


@auth.verify_opaque
def verify_opaque(opaque):
    if opaque is None or OPAQUE is None:
        return False
    return hmac.compare_digest(opaque, OPAQUE)


@bp.route('/api/v1/config/<config>/<section>/<option>', methods=['PUT'])
@auth.login_required
def handleSetUciValueRequest(config, section, option):
    try:
        data = json.loads(request.data)
        if 'value' in data.keys():
            value = data['value']
        else:
            return Response(status=400)
    except Exception as e:
        log.error(e)
        return Response(status=400)
    uci = config + "." + section + "." + option
    if not validateUciValue(uci, value):
        return Response("Value {} for {} is invalid".format(value, uci), status=400)
    uciCli, _, error = setUci(config, section, option, value)
    if error == ALL_OK:
        # commit changes
        error = commitUcis([config], uciCli)
        log.debug(f"commitUcis response: {error} | UCI: {uci}")
        if error == ALL_OK:

            # Filter UCIs that take effect immediately
            if ((config == "epic"  and section == "main"   and option == "volume") or
              (config == "epic"  and section == "snmp"   and option == "enable") or
              (config == "epic"  and section == "main"   and option == "device_alias") or
              is_ae_uci_change_immediately(config, section, option, DEVICE_MODEL)):

                if config == "epic":
                    result, error = internal_funcs.createListOfServicesToRestart([(config, section, option)])
                else:
                    result, error = addServiceToRestart(["ae_uci_reload_settings"])

                if error != ALL_OK:
                    return Response(status=500)
                t = threading.Thread(target=internal_funcs.restartServices)
                t.start()
            else:
                log.debug(f"No restart services required after change value: {value} | Device_model: {DEVICE_MODEL}")

            return Response(status=200)
        else:
            return Response(status=500)
    elif error == INVALID_PARAMETER_ERROR_CODE:
        return Response(status=404)
    else:  # EXCEPTION_OCCURRED_ERROR_CODE
        return Response(status=500)


@bp.route('/api/v1/config/<config>/<section>/<option>', methods=['GET'])
@auth.login_required
def handleGetUciValueRequest(config, section, option):
    # 400?
    # 404 -> incomplete path
    value, error = getUciValue(config, section, option)
    if error == ALL_OK:
        if value is not None:
            responseHeaders = werkzeug.datastructures.Headers()
            responseHeaders.add('Content-Type', 'application/json')
            return Response(json.dumps({'value': value}), status=200, headers=responseHeaders)
    else: # exception occurs
        return Response(status=500)

@bp.route('/api/v1/restart', methods=['PUT'])
@auth.login_required
def handleRestartRequest():
    # 400 and 404?
    error_code = reboot()
    return Response(status=error_code)

@bp.route('/api/v1/settings', methods=['GET'])
@auth.login_required
def handleExportSettingsRequest():
    settingsDict = {'version': 1, 'settings': {}}
    configsList = getAllConfigs()
    for config in configsList:
        value, error = getUciValue(config)
        if error == ALL_OK:
            if value is None:
                return Response(status=500)
            else:
                settingsDict['settings'][config] = value
        else:
            return Response(status=500)
    #log.debug(settingsDict)
    responseHeaders = werkzeug.datastructures.Headers()
    responseHeaders.add('Content-Type', 'application/json')
    return Response(json.dumps(settingsDict), status=200, headers=responseHeaders)


@bp.route('/api/v1/settings', methods=['PUT'])
@auth.login_required
def handleImportSettingsRequest():
    try:
        data = json.loads(request.data)
        if 'version' not in data.keys() or data['version'] != 1:
            return Response(status=400)
        else:
            if 'settings' not in data.keys():
                return Response(status=400)
            else:
                for config in data['settings'].keys():
                    for section in data['settings'][config].keys():
                        for option in data['settings'][config][section].keys():
                            uci = config+"."+section+"."+option
                            value = data['settings'][config][section][option]
                            if not validateUciValue(uci, value):
                                return Response(json.dumps("Value {} for {} is invalid".format(value, uci)), status=400)
                            uciCli, _, error = setUci(config, section, option, value)
                            if error == ALL_OK:
                                # commit changes
                                error = commitUcis([config], uciCli)
                                if error != ALL_OK:
                                    return Response(status=500)
                                # else continue
                            elif error == INVALID_PARAMETER_ERROR_CODE:
                                return Response(status=400)
                            else:  # EXCEPTION_OCCURRED_ERROR_CODE
                                return Response(status=500)
    except Exception as e:
        log.error(e)
        return Response(status=400)
    return Response(status=200)


@bp.route('/api/v1/password/<user>', methods=['PUT'])
@auth.login_required
def handleChangePasswordRequest(user):
    if user != "admin":
        return Response(status=404)
    try:
        data = json.loads(request.data)
    except Exception as e:
        log.error(e)
        return Response(status=400)
    if 'oldPassword' in data.keys() and 'newPassword' in data.keys():
        oldPwd = data['oldPassword']
        newPwd = data['newPassword']
        # if webui password is set
        webui_pass_flag = getUci('httpd.webserver.password_set')
        if webui_pass_flag == "true":
            if oldPwd == '':
                return Response(status=400)
            else:
                result, error = currPasswordIsValid(oldPwd)
                if error != ALL_OK:
                    return Response(status=500)
                else:
                    if result:  # password is valid
                        if newPwd != '':  # webui_pass_flag == "true" and pwd_old != '' and pwd_set != ''
                            result = setNewWebUIpassword(newPwd)
                        else:  # webui_pass_flag == "true" and pwd_old != '' and pwd_set == ''
                            result = enableOrdisableWebUIpassword('false')
                    else: # password is invalid
                        return Response(status=400)
        else:  # webui_pass_flag == "false"
            result = setNewWebUIpassword(newPwd)
        if result != ALL_OK:
            return Response(status=500)
        else:
            return Response(status=200)
    else:
        return Response(status=400)


@bp.route('/api/v1/reload-settings', methods=['PUT'])
@auth.login_required
def handleReloadSettingsRequest():
    try:
        redirectionURL = "http://127.0.0.1:48081/api/v1/reload-settings"
        r = urllib.request.Request(url=redirectionURL, method='PUT')
        resp = urllib.request.urlopen(r)
        return Response(status=resp.status)
    except Exception as e:
        log.error(e)
        return Response(status=500)


@bp.route('/api/v1/reset', methods=['PUT'])
@auth.login_required
def handleResetRequest():
    _, status_code = resetToDefaults()
    return Response(status=status_code)


@bp.route('/api/v1/dropbear', methods=['PUT'])
@auth.login_required
def handleEnableOrDisableDropbear():
    try:
        data = json.loads(request.data)
        if 'value' in data.keys():
            value = data['value']
        else:
            return Response(status=400)
    except Exception as e:
        log.error(e)
        return Response(status=400)

    uci = "dropbear.RunCtl.enable"
    if not validateUciValue(uci, value):
        return Response("Value {} for {} is invalid".format(value, uci), status=400)

    error, reason, ucisConfigured = setNewUciConfigs({uci: value}, True, True, None)
    if error == EXCEPTION_OCCURRED_ERROR_CODE:
        return Response(status=INTERNAL_SERVER_ERROR_CODE)
    elif error == INVALID_PARAMETER_ERROR_CODE:
        return_msg = {"error_code": INVALID_PARAMETER_ERROR_CODE,
                      "msg": INVALID_PARAMETER_ERROR_MSG.format(reason)}
        log.error(return_msg)
        return Response(json.dumps(return_msg), status=BAD_REQUEST_ERROR_CODE)
    else:
        return Response(status=200)
