#import json
import re
import subprocess
import logging
import tempfile
from os import listdir, mkdir, remove
from os.path import isfile, join, exists, isdir
from pathlib import Path
import threading
import shlex
import time


from werkzeug.utils import secure_filename

from .common import *

#from .uci import *

from flask import (
    Blueprint, flash, g, redirect, render_template, request, session, url_for, Response, json, make_response
)

from .internal_funcs import login_required, writeInFile, reboot, resetToDefaults

bp = Blueprint('system_funcs_API', __name__)

log = logging.getLogger('werkzeug')

updateStatus = "finished" # uploading, installing, rebooting, finished, error

@bp.route('/rest/reboot', methods=['GET'])
@login_required
def rebootSystem():
    error_code = reboot()
    return Response(status=error_code)

@bp.route('/rest/reset', methods=['GET'])
@login_required
def softReset():
    return_msg, status_code = resetToDefaults()
    return Response(json.dumps(return_msg), status=status_code)

@bp.route('/rest/system/update', methods=['POST'])
@login_required
def uploadFWImage():
    global updateStatus
    log.info("Uploading fw image...")
    try:
        updateStatus = "uploading"
        #log.debug(request.files)
        if 'fwImage' not in request.files:
            # remove file if error occurs
            removeUpdateFileChecker()
            return Response(json.dumps('No file'), status=400)
        uploadedFile = request.files['fwImage']
        #log.debug(uploadedFile)
        # if user does not select file, browser also
        # submit an empty part without filename
        if uploadedFile.filename == '':
            # remove file if error occurs
            removeUpdateFileChecker()
            return Response(json.dumps('No file selected'), status=400)
        # generate uuid and store it in file
        # if uuid exists, stop update request because there's already an update running
        # file is automatically removed after reboot
        if not isfile(FW_UPDATE_FILE):
            uid = uuid.uuid4().hex
            writeInFile(FW_UPDATE_FILE, uid)
        else:
            log.error("ABORTING UPDATE! There is an update already running.")
            return Response(json.dumps("Update already running."), status=409)
        filename = secure_filename(uploadedFile.filename)
        try:
            log.info("Storing fw update file in {}".format(UPDATES_FOLDER))
            uploadedFile.save(join(UPDATES_FOLDER, filename))
        except Exception as e:
            log.error(e)
            updateStatus = "error"
            # remove file if error occurs
            removeUpdateFileChecker()
            return Response(status=500)
        t = threading.Thread(target=installNewFW, args=(join(UPDATES_FOLDER, secure_filename(uploadedFile.filename)),))
        t.start()
    except Exception as e:
        log.error("Error occurred while uploading fw file : {}".format(e))
        updateStatus = "error"
        # remove file if error occurs
        removeUpdateFileChecker()
        if str(e).startswith('413 Request Entity Too Large'):
            return Response('REQUEST ENTITY TOO LARGE', status=413)
        else:
            return Response(status=400)
    return Response(status=200)

def installNewFW(filePath):
    global updateStatus
    updateStatus = "installing"
    cmd = "qiba-update-client -f "+filePath
    log.debug("Executing: {}".format(cmd))
    proc = None
    try:
        proc = subprocess.Popen(shlex.split(cmd), shell=False)
        proc.wait()
    except Exception as e:
        log.error(e)
        updateStatus = "error"
        # remove file if error occurs
        removeUpdateFileChecker()
    log.debug("qiba-update-client returned {}".format(proc.returncode))
    if proc.returncode != 0:
        log.error("qiba-update-client failed installation")
        updateStatus = "error"
        removeUpdateFileChecker()
        # remove file
        try:
            remove(filePath)
        except Exception as e:
            log.error("Can't remove file {}: {}".format(filePath, e))
            # remove file if error occurs
            removeUpdateFileChecker()
    else:
        # remove file after installing it
        try:
            remove(filePath)
        except Exception as e:
            log.error("Can't remove file {}: {}".format(filePath, e))
            updateStatus = "error"
            # remove file if error occurs
            removeUpdateFileChecker()
        updateStatus = "rebooting"
        time.sleep(3)
        cmd = "reboot"
        log.debug("Executing: {}".format(cmd))
        subprocess.Popen(shlex.split(cmd), shell=False)
        # no need to removeUpdateFileChecker because device will reboot and /tmp will be cleaned

def removeUpdateFileChecker():
    try:
        remove(FW_UPDATE_FILE)
    except Exception as e:
        log.error("Can't remove file {}: {}".format(FW_UPDATE_FILE, e))

@bp.route('/rest/system/update', methods=['GET'])
@login_required
def getUpdateStatus():
    log.debug(updateStatus)
    return Response(json.dumps(updateStatus),status=200)