import logging
import json
import subprocess
import time
import psutil
import datetime
from os import path
from pathlib import Path
import string
import random

from barix import HttpMethod

from uci import Uci, UciExceptionNotFound, UciException
from .uci import *

from .internal_funcs import *

from flask import (
    Blueprint, flash, g, redirect, render_template, request, session, url_for, Response, send_file, make_response,
    send_from_directory)

bp = Blueprint('system_info', __name__)

log = logging.getLogger('werkzeug')

@bp.route('/rest/info/header', methods=['GET'])
@login_required
def getHeaderSystemInfo():
    # get FW version
    fw_version = getFWVer()

    # get device type
    device_type = getDeviceType()

    # get mac address
    mac_addr = getMACaddr()

    if device_type != "" and mac_addr != "" and fw_version != "":
        return_msg=device_type+"\xa0\xa0MAC "+mac_addr+"\xa0FW "+fw_version
        return Response(return_msg, status=200)
    else:
        return Response(status=500)

@bp.route('/rest/info/firmware', methods=['GET'])
@login_required
def getCurrentFirmwareInfo():
    return_msg = {}

    #get fwVersion
    fw_version = getFWVer()
    return_msg["fw_version"] = fw_version

    #get rootFSDate
    rootFSDate = "built "
    rootFSDateFile = "/barix/info/DATE"
    try:
        with open(rootFSDateFile) as file:
            rootFSDate = rootFSDate + file.read().strip('\n')
            log.debug("RootFSDate: {}".format(rootFSDate))
            return_msg["rootFSDate"] = rootFSDate
    except Exception as e:
        log.exception("Exception found: {}".format(e))

    # get webUIVer
    webUIVer = ""
    webUIVerFile = "/usr/local/www/sys/VERSION"
    try:
        with open(webUIVerFile) as file:
            webUIVer = file.readline().strip('\n')
            log.debug("webUIVer: {}".format(webUIVer))
            return_msg["webUIVer"] = webUIVer
    except Exception as e:
        log.exception("Exception found: {}".format(e))

    # get kernelVer
    kernelVer = getKernelVer()
    return_msg["kernelVer"] = kernelVer

    # get bootloaderVer
    bootloaderVer = getBootloaderVer()
    return_msg["bootloaderVer"] = bootloaderVer

    return Response(json.dumps(return_msg), status=200)

@bp.route('/rest/info/system', methods=['GET'])
@login_required
def getSystemStatus():
    print(session)
    return_msg = {}

    # get HW_type
    dev_type = getDeviceType()
    dev_type_id = getDeviceTypeID()
    return_msg["HW_type"] = dev_type+" (ID "+dev_type_id+")"

    # get IPAM_type
    IPAM_type = getIPAMType()
    IPAM_type_id = getIPAMTypeID()
    return_msg["IPAM_type"] = IPAM_type + " (ID " + IPAM_type_id + ")"

    # get mac address
    mac_addr = getMACaddr()
    return_msg["mac_addr"] = mac_addr

    # get IP_addr
    ip_addr = getIPaddr('eth0')
    return_msg["ip_addr"] = ip_addr

    # get netmask
    netmask = getNetmask()
    return_msg["netmask"] = netmask

    # get defaultGW
    defaultGW = getDefaultGW()
    return_msg["default_gateway"] = defaultGW

    # get DNS_servers
    dns_servers = getDnsServers()
    return_msg["DSN_servers"] = dns_servers

    # get FW
    return_msg["FW"] = getAppNameFromManifest()

    # get fwVersion
    fw_version = getFWVer()
    return_msg["fw_version"] = fw_version

    # get Application
    return_msg["app"] = getApplication()

    # get Application version
    return_msg["app_ver"] = getAppVersion()

    # get kernelVer
    kernelVer = getKernelVer()
    return_msg["kernelVer"] = kernelVer

    # get bootloaderVer
    bootloaderVer = getBootloaderVer()
    return_msg["bootloaderVer"] = bootloaderVer

    # get system time
    return_msg["sys_time"] = getSystemTime()

    # get uptime
    return_msg["uptime"] = getUpTime()

    # get lic data
    lic_dict = getLicenses()
    for key in lic_dict:
        return_msg[key] = lic_dict[key]

    # get storage devices
    return_msg["massStorageDev"] = getMassStorageDevices()

    # get HW Type ID
    return_msg["HW_type_id"] = getHWTypeID()

    # get PoE Power Source
    return_msg["poe_power_source"] = getPoEConfig()

    # get PoE Speaker Impedance
    return_msg["poe_speaker_impedance"] = getPoESpkImp()

    # get PoE Amplifier Gain
    return_msg["poe_amp_gain"] = getPoEAmpGain()

    # get Amplifier over temperature status
    return_msg["otw_status"] = getPoEOverTempStatus()

    log.debug(return_msg)
    return Response(json.dumps(return_msg), status=200)

@bp.route('/rest/system/info', methods=['GET'])
@login_required
def getSystemInfo():
    log.debug(request.args)

    settingsRequested = request.args
    result_dict = {}

    for requestedSetting in settingsRequested:
        if requestedSetting == "device_type":
            # get HW_type
            dev_type = getDeviceType()
            result_dict["device_type"] = dev_type

        elif requestedSetting == "device_type_ID":
            # get HW_type_id
            dev_type_id = getDeviceTypeID()
            result_dict["device_type_ID"] = dev_type_id

        elif requestedSetting == "IPAM_type":
            # get IPAM_type
            IPAM_type = getIPAMType()
            result_dict["IPAM_type"] = IPAM_type

        elif requestedSetting == "IPAM_type_ID":
            # get IPAM_type_id
            IPAM_type_id = getIPAMTypeID()
            result_dict["IPAM_type_ID"] = IPAM_type_id

        elif requestedSetting == "mac_addr":
            # get mac address
            mac_addr = getMACaddr()
            result_dict["mac_addr"] = mac_addr

        elif requestedSetting == "ip_addr":
            # get IP_addr
            ip_addr = getIPaddr('eth0')
            result_dict["ip_addr"] = ip_addr

        elif requestedSetting == "netmask":
            # get netmask
            netmask = getNetmask()
            result_dict["netmask"] = netmask

        elif requestedSetting == "default_gateway":
            # get defaultGW
            defaultGW = getDefaultGW()
            result_dict["default_gateway"] = defaultGW

        elif requestedSetting == "DSN_servers":
            # get DNS_servers
            dns_servers = getDnsServers()
            result_dict["DSN_servers"] = dns_servers

        elif requestedSetting == "FW":
            # get FW
            #TODO: get from manifest file
            result_dict["FW"] = "IP Speaker"

        elif requestedSetting == "fw_version":
            # get fwVersion
            fw_version = getFWVer()
            result_dict["fw_version"] = fw_version

        elif requestedSetting == "app":
            # get Application
            result_dict["app"] = getApplication()

        elif requestedSetting == "app_ver":
            # get Application version
            result_dict["app_ver"] = getAppVersion()

        elif requestedSetting == "kernelVer":
            # get kernelVer
            kernelVer = getKernelVer()
            result_dict["kernelVer"] = kernelVer

        elif requestedSetting == "bootloaderVer":
            # get bootloaderVer
            bootloaderVer = getBootloaderVer()
            result_dict["bootloaderVer"] = bootloaderVer

        elif requestedSetting == "sys_time":
            # get system time
            result_dict["sys_time"] = getSystemTime()

        elif requestedSetting == "uptime":
            # get uptime
            result_dict["uptime"] = getUpTime()

        elif requestedSetting == "license":
            # get lic data
            result_dict['licenses'] = getLicenses()

        elif requestedSetting == "massStorageDev":
            # get storage devices
            result_dict["massStorageDev"] = getMassStorageDevices()

        elif requestedSetting == "HW_type_ID":
            # get HW Type ID
            result_dict["HW_type_ID"] = getHWTypeID()

        elif requestedSetting == "conn_status":
            result_dict["conn_status"] = getWiredConnStatus()

        elif requestedSetting == "device_model":
            result_dict["device_model"] = getDeviceModel()

        elif requestedSetting == "open_source_licenses":
            result_dict["open_source_licenses"] = getOpenSourceLicenses()

        else:
            log.debug("Unrecognized setting {}".format(requestedSetting))
            return Response("Unrecognized setting {}".format(requestedSetting), status=400)

    log.debug(result_dict)
    return Response(json.dumps(result_dict), status=200)

@bp.route('/rest/system/type', methods=['GET'])
@login_required
def getSystemType():
    return_msg = {}

    return_msg['HW_type_ID'] = getHWTypeID()

    if path.exists('/etc/barix/ussi/BARIX_USSI.json'):
        return_msg['hasSNMPSettings'] = True

    if path.exists('/tmp/poe_detect'):
        return_msg['hasPoESettings'] = True

    if path.exists('/tmp/otw_status'):
        f = open("/tmp/otw_status", "r")
        return_msg['otw_status'] = f.read()
        f.close()

    log.debug(return_msg)
    return Response(json.dumps(return_msg), status=200)

@bp.route('/rest/system/cert', methods=['GET'])
@login_required
def getUploadedCertificates():
    certificateList = getCertificates()
    if certificateList is None:
        return Response(status=500)
    return_dict = {"custom" : certificateList}
    return Response(json.dumps(return_dict), status=200)

@bp.route('/rest/system/cert', methods=['POST'])
@login_required
def uploadCertificate():
    log.debug("Certificate uploaded")
    log.debug(request.files)
    if 'certificate' not in request.files:
        return Response(json.dumps('No file'), status=400)
    uploadedFile = request.files['certificate']
    log.debug(uploadedFile)
    # if user does not select file, browser also
    # submit an empty part without filename
    if uploadedFile.filename == '':
        return Response(json.dumps('No file selected'), status=400)
    try:
        # check file size
        # generate random string
        letters = string.ascii_lowercase
        tmp_filename = ''.join(random.choice(letters) for i in range(8))
        tmp_path = os.path.join('/tmp', tmp_filename)

        uploadedFile.save(tmp_path)
        filesize = os.stat(tmp_path).st_size
        print(filesize)
        if filesize < 133 or filesize > 16384000:
            return Response(json.dumps("File size is invalid. Please make sure you're uploading the right file."), status=400)
    except Exception as e:
        log.error(e)
    # check file extension
    ALLOWED_EXTENSIONS = {'crt'}
    if '.' in uploadedFile.filename and uploadedFile.filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS:
        res = addCertificates(tmp_path, secure_filename(uploadedFile.filename))
        if res is None:
            msg = uploadedFile.filename + " is not a valid certificate. Not installed."
            return Response(json.dumps(msg), status=400)
        elif res:
            msg = uploadedFile.filename + " installed."
            return Response(json.dumps(msg), status=200)
        else:
            msg = uploadedFile.filename + " installation failed."
            return Response(json.dumps(msg), status=400)
    else:
        return Response(json.dumps('Format not supported'), status=400)


@bp.route('/rest/system/cert', methods=['DELETE'])
@login_required
def deleteCertificates():
    print(request.data)
    try:
        data = json.loads(request.data)
    except Exception as e:
        log.error(e)
        return Response(status=400)
    res = removeCertificates(data['payload'])
    return Response(json.dumps(res), status=200)

@bp.route('/rest/logs/system', methods=['GET'])
@login_required
def getSystemLog():
    return_msg = getLastFileLines('/var/log/messages', 50)
    return Response(return_msg, status=200)

@bp.route('/rest/logs', methods=['GET'])
@login_required
def downloadAllLogs():
    appName = getAppNameFromManifest()
    macAddr = getMACaddr()
    timestamp = datetime.datetime.now().strftime("%Y%m%d%H%M%S")
    filename = appName + "_" + macAddr + "_" + timestamp + "_logs.tar.gz"
    #
    # # remove any previous log that may exist
    # try:
    #     my_dir = '/mnt/data/' # enter the dir name
    #     for fname in os.listdir(my_dir):
    #         if fname.startswith(appName + "_" + macAddr + "_"):
    #             os.remove(os.path.join(my_dir, fname))
    # except Exception as e:
    #     log.error(e)
    #
    # proc = subprocess.run(['uci', 'show'], capture_output=True)
    # uciListToWrite = proc.stdout.decode("utf-8")
    # try:
    #     f = open("/var/log/uci.config", "w")
    #     f.write(uciListToWrite)
    #     f.close()
    # except Exception as e:
    #     log.error("Error creating /var/log/uci.config file: {}".format(e))
    #     return Response(status=500)
    #
    # try:
    #     cmd = "tar -czf /mnt/data/" + filename + " -C /var/log ."
    #     subprocess.run(shlex.split(cmd))
    # except Exception as e:
    #     log.error(e)
    #
    # return send_file(os.path.join('/mnt/data', filename), as_attachment=True, attachment_filename=filename, cache_timeout=0)
    try:
        cmd = "/bin/sh /usr/local/bin/collect-logs.sh"
        proc = subprocess.Popen(shlex.split(cmd))
        proc.wait()
        if proc.returncode != 0:
            return Response(status=500)
    except Exception as e:
        log.error(e)
        return Response(status=500)
    return send_file('/mnt/data/last_logs.tar.gz', as_attachment=True, download_name=filename,
                     max_age=0)
