local dbus = require("lua-dbus");
local interface = require("interfaces.interface");
local dev_man = require("arcanix.device_manager");
local logging = require("logging");
local logger = logging.defaultLogger();

local module = {};

local uv;

local BLUEZ_DEV_PATH = '/org/bluez/hci0';

local function findBleAdapter(cbk)
    dbus.call("GetManagedObjects", function(...)
        -- find an object with GattManager support
        for obj_path,ifaces in pairs(...) do
            for iface_name, _ in pairs(ifaces) do
                if iface_name == "org.bluez.GattManager1" then
                    logger:info("Found BLE adapter: %s", obj_path);
                    BLUEZ_DEV_PATH = obj_path;
                    cbk();
                    return;
                end
            end
        end
        logger:error("BLE adapter not found!!");
    end,
    {
        type = "method_call",
        bus = "system",
        path = "/",
        interface = "org.freedesktop.DBus.ObjectManager",
        destination = "org.bluez",
    });
end

local function registerAgent(cbk)
    module.ifaceAgent = interface.new("/org/bluez/agent", "org.bluez.Agent1", nil, "system");

    local res = dbus.call('RegisterAgent', function (...)
        logger:debug("Agent registered");
        cbk();
    end, {
        bus = 'system',
        path = '/org/bluez', -- change this!
        interface = "org.bluez.AgentManager1",
        destination = "org.bluez",
        args = {
            "o",
            "/org/bluez/agent",
            "s",
            ""
        }
    });

end

local function setPower(power, cbk)

    local res = dbus.call('Set', function (...)
        logger:debug("Power set");
        cbk();
    end, {
        bus = 'system',
        path = BLUEZ_DEV_PATH,
        interface = "org.freedesktop.DBus.Properties",
        destination = "org.bluez",
        args = {
            "s",
            "org.bluez.Adapter1",
            "s",
            "Powered",
            "v",
            {
                typ = "b",
                value = power
            }
        }
    });
end

local function registerApp(cbk)

    module.ifaceProfile = interface.new("/org/bluez/app", "org.bluez.GattProfile1", {
        ["UUIDs"] = {
            typ = "as",
            value = {
                "register-application"
            }
        }
    }, "system");

    local res = dbus.call('RegisterApplication', function (...)
        cbk();
    end, {
        bus = 'system',
        path = BLUEZ_DEV_PATH,
        interface = "org.bluez.GattManager1",
        destination = "org.bluez",
        args = {
            "o",
            "/",
            "a{sv}",
            {}
        }
    });

end

local function registerAdvertise(cbk, name)
    module.ifaceAdvertise = interface.new("/org/bluez/advertising1", "org.bluez.LEAdvertisement1", {
        ["Type"] = {
            typ = "s",
            value = "peripheral"
        },
        ["Discoverable"] = {
            typ = "b",
            value = true
        },
        ["LocalName"] = {
            typ = "s",
            value = name
        },
    }, "system", {
        ["Release"] = function()
            logger:info("Advertise unregistered!")
        end
    });

    local res = dbus.call('RegisterAdvertisement', function (...)
        cbk();
    end, {
        bus = 'system',
        path = BLUEZ_DEV_PATH,
        interface = "org.bluez.LEAdvertisingManager1",
        destination = "org.bluez",
        args = {
            "o",
            "/org/bluez/advertising1",
            "a{sv}",
            {}
        }
    });
end


local function unregisterAdvertise(cbk)
    local res = dbus.call('UnregisterAdvertisement', function (...)
        cbk()
    end, {
        bus = 'system',
        path = BLUEZ_DEV_PATH,
        interface = "org.bluez.LEAdvertisingManager1",
        destination = "org.bluez",
        args = {
            "o",
            "/org/bluez/advertising1"
        }
    });
end


function module.init(luv, cbk)
    uv = luv;
    local ldbus = dbus.init();
    interface.init(dbus);

    findBleAdapter(function()
        registerAgent(function()

            setPower(true, cbk);

            -- this is to track the connected and disconnected devices
            dbus.on("PropertiesChanged", function(path, iface_name, changed_prop, ...) 
                if type(iface_name) == "string" and iface_name == "org.bluez.Device1" then
                    if changed_prop.ServicesResolved ~= nil then
                        if changed_prop.ServicesResolved then
                            logger:info("Device %s has connected", path)
                            dev_man.add_device(path);
                        else
                            logger:info("Device %s has disconnected", path)
                            dev_man.remove_device(path);
                        end
                    end
                end
            end, {
                type="signal",
                bus = "system",
                interface = "org.freedesktop.DBus.Properties",
                path_keyword = "path"
            });
        end);
    end);

    module.pollTimer = uv.new_timer();
    module.pollTimer:start(50, 50, dbus.poll);
    module.serviceCounter = 0;
end

function module.addService(uuid, primary)
    local serviceId = module.serviceCounter;
    module.serviceCounter = module.serviceCounter + 1;

    local iface = interface.new("/org/bluez/app/service"..tostring(serviceId), "org.bluez.GattService1", {
        ["Handle"] = {
            typ = "q",
            value = 0
        },
        ["UUID"] = {
            typ = "s",
            value = uuid
        },
        ["Primary"] = {
            typ = "b",
            value = primary
        }
    }, "system");

    local obj = {
    	characteristics = {},
    	interface = iface,
    	serviceId = serviceId,
    	charCounter = 0
    };

    obj.addCharacteristic = function(charUuid, value, flags, onRead, onWrite)

    	if obj.characteristics[charUuid]~=nil then
    		return false;
    	end

    	local charId = obj.charCounter;
    	obj.charCounter = obj.charCounter + 1;

    	local valueArr = {};

    	for i=1,#value,1 do
    		valueArr[i] = value:byte(i);
    	end

    	local ifaceChar;
	    ifaceChar = interface.new("/org/bluez/app/service"..tostring(serviceId).."/chrc"..tostring(charId), "org.bluez.GattCharacteristic1", {
	        ["Handle"] = {
	            typ = "q",
	            value = 0
	        },
	        ["UUID"] = {
	            typ = "s",
	            value = charUuid
	        },
	        ["Service"] = {
	            typ = "o",
	            value = "/org/bluez/app/service"..tostring(serviceId)
	        },
	        ["Value"] = {
	            typ = "ay",
	            value = valueArr
	        },
	        ["Notifying"] = {
	            typ = "b",
	            value = false
	        },
	        ["Flags"] = {
	            typ = "as",
	            value = flags
	        }
	    }, "system", {
	        ["ReadValue"] = function(reqObj, options)
                if onRead~=nil then
                    local device = dev_man.find(options.device);
                    if device == nil then
                        return "ay", nil
                    end

	            	local respStr = onRead(device);

			    	local respBytes = {};

                    if respStr~=nil then
    			    	for i=1,#respStr,1 do
    			    		respBytes[i] = respStr:byte(i);
    			    	end
                    end

			    	ifaceChar.properties["Value"].value = respBytes;

			    	return "ay", respBytes;
	            end

	            return "ay", ifaceChar.properties["Value"].value;
	        end,
	        ["WriteValue"] = function(reqObj, value, options)
	            for i=1,#value,1 do
	                value[i] = string.char(value[i]);
	            end
	            local str = table.concat(value);
	            if onWrite~=nil then
                    local device = dev_man.find(options.device);
                    if device == nil then
                        return;
                    end
	            	onWrite(str, device);
	            end
	        end,
	        ["StartNotify"] = function(reqObj)
	        	logger:debug("Characteristic StartNotify");
	        	ifaceChar.properties["Notifying"].value = true;
		    end,
		    ["StopNotify"] = function(reqObj)
                logger:debug("Characteristic StopNotify");
                ifaceChar.properties["Notifying"].value = false;
		    end
	    });

	    obj.characteristics[charUuid] = {
	    	interface = ifaceChar,
	    	charId = charId,
	    	set = function(str)
		    	local respBytes = {};

		    	for i=1,#str,1 do
		    		respBytes[i] = str:byte(i);
		    	end
				ifaceChar.properties["Value"].value = respBytes;
				ifaceChar.properties["Value"].notifyChanged();
	    	end
	    }

	    return obj.characteristics[charUuid];

	end;

	return obj;

end

function module.start(name, cbk)
    registerApp(function()
        registerAdvertise(cbk, name);
    end);
end

function module.stop()
    unregisterAdvertise(function()
        print("Advertise unregistered")
    end)
    setPower(false, function()
        print("Powered off")
    end)
    module.pollTimer:stop()
end

return module;
