local ble = require("arcanix.ble");
local md5 = require("arcanix.md5");
local str_utils = require("arcanix.str_utils");
local dev_man = require("arcanix.device_manager");
local logging = require("logging");
local cjson = require("cjson")
local logger = logging.defaultLogger();
require("arcanix.configuration_handler")
require("arcanix.platform");

local uv;

local module = {};

-- Current Arcanix version implementation
local PROTO_VERSION = 1;

local USE_KEY = "9jasdoi1 nbhsa129sad ads55523";

local BASE_UUID = "-0000-1000-8000-0242ac120002";

local ARCANIX_UUID = {
    service="00000000"..BASE_UUID,
    regid="00000001"..BASE_UUID,
    challenge="00000002"..BASE_UUID,
    authentication="00000003"..BASE_UUID,
    pin_value="00000004"..BASE_UUID,
    conf_submit="00000005"..BASE_UUID,
    application="00000006"..BASE_UUID,
    protoVersion="00000007"..BASE_UUID,
    deviceType="00000008"..BASE_UUID,
    softwareVersion="00000009"..BASE_UUID,
    macAddr="0000000A"..BASE_UUID
}

-- Abstraction for the platform
local myPlatform;

local myConfHandler;

-- Called when app writes on application characteristic
local app_callback;

local arcanix = {
    auth_mode=AUTH_MODE.PIN,
    auth_algorithm=AUTH_ALGO.MD5,
    pin_value="0000"
};

local function binaryToHex(str)
    local strArr = {};
    for i=1,#str,1 do
        strArr[#strArr+1] = string.format("%02x", str:byte(i));
    end
    return table.concat(strArr);
end

local function readModel(filename)
    local f = io.open(filename, "rb")
    if f then
        local content = f:read("*a")
        f:close()
        return content
    end

    return nil
end

local function startBLEServer(regId, name, deviceType)
    logger:info("My BLE name is %s", name);

    ble.init(uv, function()

        local defaultService = ble.addService(ARCANIX_UUID.service, true);

        -- retrieve current Arcanix protocol version
        defaultService.addCharacteristic(ARCANIX_UUID.protoVersion, "\0", {"read"},
        function(device)
            return tostring(PROTO_VERSION);
        end);

        -- retrieve regId
        defaultService.addCharacteristic(ARCANIX_UUID.regid, "\0", {"read"},
        function(device)
            --device:setAuthenticated();
            return regId;
        end);

        -- retrieve device type
        defaultService.addCharacteristic(ARCANIX_UUID.deviceType, "\0", {"read"},
        function(device)
            return deviceType;
        end);

        -- retrieve software versions
        defaultService.addCharacteristic(ARCANIX_UUID.softwareVersion, "\0", {"read"},
        function(device)
            local versionsAnswer = { fw=myPlatform:getFwVersion() }
            local appVersion = myPlatform:getAppVersion()
            if appVersion ~= nil then
                versionsAnswer["app"] = appVersion
            end
            return cjson.encode(versionsAnswer)
        end);

        -- read MAC address
        defaultService.addCharacteristic(ARCANIX_UUID.macAddr, "\0", {"read"},
        function(device)
            return myPlatform:getMacAddress()
        end);

        -- Generates a challenge to be used on authentication
        defaultService.addCharacteristic(ARCANIX_UUID.challenge, "\0", {"read"},
        function(device)
            math.randomseed(os.time()+device.lastRandom);
            device.lastRandom = math.random(0, 2^32);
            device.challenge = md5.sum(device.name..regId..device.lastRandom);
            logger:debug("device %s new challenge %s", device.name, binaryToHex(device.challenge));
            -- Challenge message format;
            -- Version, N Key Modes, List of Key Modes, N Key Algorithm, List of Key Algorithm, The challenge
            return "1,1,1,1,1,"..device.challenge;
        end);

        -- Device authentication
        defaultService.addCharacteristic(ARCANIX_UUID.authentication, "\0", {"read", "write"},
        function(device)
            return tostring(device.authenticated);
        end, function(value, device)
            if device.challenge~=nil then
                logger:debug("authentication message %s", binaryToHex(value));
                local split = str_utils.split(value, ",");
                -- For now only version 1 is supported
                if #split == 4 and split[1] == "1" then
                    local auth_mode = tonumber(split[2]);
                    local auth_algorithm = tonumber(split[3]);
                    local hash = split[4];
                    if auth_mode == arcanix.auth_mode and auth_algorithm == arcanix.auth_algorithm then
                        local desired = md5.sum(device.challenge..arcanix.pin_value..USE_KEY);
                        logger:debug("Calculating auth hash for %s", binaryToHex(device.challenge..arcanix.pin_value..USE_KEY));
                        logger:debug("Hash received: %s, Desired: %s", binaryToHex(hash), binaryToHex(desired));
                        if hash==desired and not device:isBlocked() then
                            logger:info("Device %s authenticated", device.name);
                            device:setAuthenticated();
                        else
                            device:setAuthFailed();
                        end
                    end
                end
            end
        end);

        -- PIN: value modification
        defaultService.addCharacteristic(ARCANIX_UUID.pin_value, "\0", {"write"}, nil, function(value, device)
            if device.authenticated then
                local status, pin_value = pcall(tostring, value);
                if pin_value ~= arcanix.pin_value then
                    -- modify pin value
                    arcanix.pin_value = pin_value;
                    myConfHandler:save_new_pin(pin_value);
                    logger:info("new pin value was set %s", pin_value);
                    if arcanix.auth_mode == AUTH_MODE.PIN then
                        -- a new authentication is needed
                        device.authenticated = false;
                    end
                end
            end
        end);

        -- Apply changes to the UCIs
        defaultService.addCharacteristic(ARCANIX_UUID.conf_submit, "\0", {"write"}, nil,
        function(value, device)
            if device.authenticated then
                myConfHandler:reload_settings();
            end
        end);

        -- This characteristic is processed on application level
        defaultService.addCharacteristic(ARCANIX_UUID.application, "\0", {"write"}, nil,
        function(value, device)
            if device.authenticated and app_callback ~= nil then
                app_callback(value);
            end
        end);

        -- uci configurations
        local characteristics = {};
        myConfHandler:add_ble_characteristics(function(name, uuid, confWrite)
            local full_uuid = string.format("%08X", uuid)..BASE_UUID;
            characteristics[name] = defaultService.addCharacteristic(full_uuid, "\0", {"read", "write", "notify"},
            function(device)
                if device.authenticated then
                    return tostring(myConfHandler:get_value(name));
                end
            end, function(value, device)
                if device.authenticated then
                    confWrite(value);
                end
            end);
        end);
        
        adv_name=name.." ("..deviceType..")"
        ble.start(adv_name, function()
            logger:debug("BLE started");
            myConfHandler:start_monitor(uv, characteristics);
        end);
    end);

end

function trim(s)
    return (string.gsub(s, "^%s*(.-)%s*$", "%1"))
end

local function getMAC()
    local f = io.open("/sys/class/net/eth0/address", "rb")
    if f then
        local content = f:read("*a")
        f:close()
        return trim(content)
    end

    return nil
end

function module.start(luv, platform, conf_handler, app_cbk, deviceType)
    if platform then
        myPlatform = createPlatform(platform);
    else
        -- default device
        myPlatform = createPlatform("ipam400");
    end

    if myPlatform == nil then
        error("unknown platform device")
    end

    if conf_handler == nil then
        error("configurations not initialized")
    end

    uv = luv;
    myConfHandler = conf_handler
    app_callback = app_cbk;

    myConfHandler.confDirectory = myPlatform:getConfDir()

    logger:debug("My platform is "..myPlatform:getName());
    logger:debug("My configuration directory is "..myConfHandler.confDirectory);
    if deviceType then
        logger:debug("My device type is "..deviceType);
    end

    -- read configuration for arcanix application
    local cfg = myConfHandler:read_arcarnix_conf()
    if cfg.auth_mode ~= nil then
        arcanix.auth_mode = cfg.auth_mode;
    end
    if cfg.auth_algorithm ~= nil then
        arcanix.auth_algorithm = cfg.auth_algorithm;
    end
    if cfg.pin_value ~= nil then
        arcanix.pin_value = cfg.pin_value;
    end
    for k,v in pairs(arcanix) do
        logger:debug("%s=%s", k , v);
    end
    dev_man.enable_pin_auth(arcanix.auth_mode == AUTH_MODE.PIN);

    -- read all configurations
    myConfHandler:read_configurations();

    local regId = myPlatform:getRegId();
    local mac_addr = getMAC();
    mac_addr = string.sub(mac_addr, -8);
    startBLEServer(regId, "Brx-"..mac_addr, deviceType);
end

function module.stop()
    ble.stop()
end

return module;
