--[[
  Implements an abstraction of the device.
  Call createDevice function to get your device instance.
  The following devices are currently supported:
     - emulator
     - ipam400
     - bavidi
]]

local function readBarixFwVersion()
    local fp = io.open("/barix/info/VERSION")
    if fp ~= nil then
        local version = fp:read("*a")
        fp:close()
        return version
    end
    return ""
end

local function readMacAddress()
    local fp = io.open("/sys/class/net/eth0/address")
    if fp ~= nil then
        local mac = fp:read("*a"):gsub("%\n", "")
        fp:close()
        return mac
    end
    return ""
end

-- Device/Emulator definition
local Platform = { 
    name="emulator",
    confDir="./conf",
};

function Platform:new(o)
    o = o or {}
    setmetatable(o, self);
    self.__index = function(table, key)
        return self[key]
    end
    self.regId = "emulator"
    self.fwVersion = "1.0.0"
    self.appVersion = nil
    self.macAddr = "00:01:02:03:04.05"
    return o;
end

function Platform:getRegId()
    return self.regId
end

function Platform:getName()
    return self.name;
end

function Platform:getConfDir()
    return self.confDir;
end

function Platform:getFwVersion()
    return self.fwVersion
end

function Platform:getAppVersion()
    return self.appVersion
end

function Platform:getMacAddress()
    return self.macAddr
end

-- IPAM400 definition
local IPAM400 = Platform:new();
local IPAM_CONFIG_FILE = "/mnt/shadow/config.json";

local function readIpamRegId()
    local json = require("cjson");
    local file = io.open(IPAM_CONFIG_FILE, "r");
    if file == nil then
        io.stderr:write("error opening ipam config file");
        os.exit(1);
    end
    contents = file:read("*a");
    io.close(file);
    return json.decode(contents)["registrationId"];
end


function IPAM400:new(o)
    o = o or Platform:new(o)
    setmetatable(o, self)
    self.__index = self
    self.regId = readIpamRegId()
    self.fwVersion = readBarixFwVersion()
    self.macAddr = readMacAddress()
    return o
end

-- IPAM400 definition
local Bavidi = Platform:new();

local function readBavidiRegId()
    local fp = io.popen("/usr/bin/bavidi-read-prod-info.sh Device.RegId")
    local regId = fp:read("*a")
    _, _, ret = fp:close()

    -- on error, we return the proc id to support devices without EEPROM
    if ret ~= 0 then
        local file = io.open("/sys/firmware/devicetree/base/serial-number", "r")
        if file ~= nil then
            local regId = file:read("*a"):gsub("%\0", "")
            io.close(file)
            return regId
        else
            return ""
        end
    end

    return regId
end

local function readBavidiAppVersion()
    local fp = io.open("/bavidi/app/version")
    if fp ~= nil then
        local version = fp:read("*a")
        fp:close()
        return version
    end
    return ""
end

function Bavidi:new(o)
    o = o or Platform:new(o)
    setmetatable(o, self)
    self.__index = self
    self.regId = readBavidiRegId()
    self.fwVersion = readBarixFwVersion()
    self.appVersion = readBavidiAppVersion()
    self.macAddr = readMacAddress()
    return o
end

function createPlatform(devName)
    if devName == "emulator" then
        return Platform:new()
    elseif devName == "ipam400" then
        return IPAM400:new{
            name = "ipam400",
            confDir = "/barix/config/current"
        }
    elseif devName == "bavidi" then
        return Bavidi:new{
            name = "bavidi",
            confDir = "/data/config"
        }
    else
        error("platform not supported!")
    end
end
