local conf_handler = require("arcanix.configuration_handler")
local str_utils = require("arcanix.str_utils")
local logging = require("logging")
local logger = logging.defaultLogger()
require("uci");

-- UCI configuration handler definition----------------------------------
local UciConfHandler = conf_handler:new()

function UciConfHandler:new(o)
    o = o or conf_handler:new(o)
    setmetatable(o, self)
    self.__index = self
    -- List of all exposed ucis
    self.uciMap = {}
    -- List of all UCIs files
    self.uciFiles = {}
    return o
end

local function uciFill(obj, values, prefix)
    if type(obj)=="table" then
        for k,v in pairs(obj) do
            if type(k)~="string" or k:byte(1)~=string.byte(".") then
                uciFill(v, values, prefix.."."..k)
            end
        end
    else
        values[prefix] = obj or ""
    end
end

function UciConfHandler:uciGetValues(base)
    local values = {}
    -- print(base)
    local uci_cursor = uci.cursor(self.confDirectory)
    -- "base" is the uci name
    local allSystem = uci_cursor:get_all(base)
    uciFill(allSystem, values, base)

    return values
end

function UciConfHandler:add_configuration(name, uuid, cbk)
    if not self:validate_uuid(uuid) then
        logger:error("invalid uuid %s", uuid)
        return
    end

    self.uciMap[name] = {
        uuid=uuid,
        cbk=cbk,
        value=""
    }
end

function UciConfHandler:read_arcarnix_conf()
    local arcanix = {}
    local cfg = self:uciGetValues("arcanix")

    if string.lower(cfg["arcanix.general.auth_mode"]) == "pin" then
        arcanix.auth_mode = AUTH_MODE.PIN
    end

    if string.lower(cfg["arcanix.general.auth_algorithm"]) == "md5" then
        arcanix.auth_algorithm = AUTH_ALGO.MD5
    end

    arcanix.pin_value = cfg["arcanix.general.pin_value"]

    return arcanix
end

function UciConfHandler:read_configurations()
    -- fill a table with all uci files present in the UCI map
    -- it will be used also to inform what uci were modified
    for u,param in pairs(self.uciMap) do
        local sections = str_utils.split(u, ".")
        if sections[1] and not self.uciFiles[sections[1]] then
            self.uciFiles[sections[1]]=true
        end

        local values = self:uciGetValues(u);
        if values[u] ~= nil then
            param.value = values[u];
        end

        logger:info("%s %s 0x%08X", u, param.value, param.uuid);

        param.write = function(toWrite)
            logger:info("New value for %s, old %s, new %s", u, param.value, toWrite);
            local uci_cursor = uci.cursor(self.confDirectory)
            if sections[3]~=nil then
                uci_cursor:set(sections[1], sections[2], sections[3], toWrite);
            else
                uci_cursor:set(sections[1], sections[2], toWrite)
            end
            param.value = toWrite
            uci_cursor:commit(sections[1])
            -- add callback to the list to be called when requested
            local add_cbk = true
            for _, cbk in pairs(self.reloadCallbacks) do
                if cbk == param.cbk then
                    add_cbk = false
                    break;
                end
            end
            if add_cbk then
                table.insert(self.reloadCallbacks, param.cbk);
            end
        end
    end
end

function UciConfHandler:start_monitor(luv, characteristics)
    local fse = luv.new_fs_event();
    fse:start(self.confDirectory, {
        watch_entry=true,
        stat=true,
        recursive=false
    }, function(err, filename, events)
        if filename~=nil and self.uciFiles[filename] then
            logger:debug("FS event: %s", filename);
            local newValues = self:uciGetValues(filename);
            for u,value in pairs(newValues) do
                if value ~= nil and self.uciMap[u] ~= nil and self.uciMap[u].value ~= value then
                    logger:info("%s has changed from %s to %s", u, self.uciMap[u].value, value);
                    self.uciMap[u].value = value;
                    if characteristics~=nil and characteristics[u]~=nil then
                        characteristics[u].set(value);
                    end
                end
            end
        end
    end);
end

function UciConfHandler:save_new_pin(pin_value)
    local u = uci.cursor(self.confDirectory)
    u:set("arcanix", "general", "pin_value", pin_value)
    u:commit("arcanix")
end

function UciConfHandler:add_ble_characteristics(callback)
    for name, param in pairs(self.uciMap) do
        callback(name, param.uuid, param.write)
    end
end

function UciConfHandler:get_value(name)
    return self.uciMap[name].value
end

return UciConfHandler
