
#  TAS2770 Digital Amplifier
#
#
#  Warning:     When audio is stopped, and there is nothing to keep the i2s interface
#               "busy" (ex like playing silent loop), so tas-2771 amp has to be initialized
#               again the next time the audio starts.

import time
from . audio_amp import *

class TAS2770(AudioAmp):

    def __init__(self, i2c_bus, chip_address, amp_vol=0x10):
        super(TAS2770, self).__init__(i2c_bus,chip_address)
        self.amp_vol = amp_vol

    """
    prepare the i2c regs to write
    We mostly write to page 0, book 0, so ignore page 1 for now
    """
    def prepareRegs(self):
        # set page 0
        self.writeByte(0x00, 0x00)
        # set book 0
        self.writeByte(0x7f, 0x00)
        # set again page 0
        self.writeByte(0x00, 0x00)

    """
    reset the TAS2770 device
    always call this function after setting the page to write and before the init
    """
    def resetAmp(self):
        self.writeByte(0x01, 0x01)
        print("Reset TAS2770 amp")

    def initAmp(self):
        self.writeByte(0x04, 0x00)
        #mute the PCM Volume at start
        self.writeByte(0x05, 0xff)

        self.writeByte(0x07, 0x02)
        self.writeByte(0x08, 0x40)
        self.writeByte(0x0a, 0x37)
        self.writeByte(0x0b, 0x03)
        self.writeByte(0x0c, 0x10)
        self.writeByte(0x0e, 0xf3)
        self.writeByte(0x15, 0x15)
        self.writeByte(0x18, 0x4b)
        self.writeByte(0x1a, 0x10)
        self.writeByte(0x32, 0x80)
        self.writeByte(0x3c, 0x18)
        self.writeByte(0x7e, 0x19)
        #set BOP enable
        self.writeByte(0x1b, 0x01)
        self.writeByte(0x1c, 0xff)
        #setting amplifier volume  
        print("Will Set Amp volume to: {}".format(hex(self.amp_vol)))
        self.writeByte(0x03, self.amp_vol)
        # set pcm_volume
        self.writeByte(0x05, 0x00)
        print("TAS2770 Initialized!")

    def powerOnAmp(self):
        self.writeByte(0x02, 0x0c)
        print("TAS2770 amplifier powered on!")

    def setupAmp(self):
        self.prepareRegs()
        self.resetAmp()
        time.sleep(1)
        self.initAmp()
        time.sleep(0.002)
        self.powerOnAmp()

    def setVol(self, vol):
        print("Will Set Amp volume to: {}".format(hex(vol)))
        self.writeByte(0x05, vol)
        self.amp_vol = vol

    def getVol(self):
        return hex(self.readByte(0x05))

    def setGain(self, gain=None):
        if gain is not None:
            self.gain = gain
        self.writeByte(0x03, self.gain)
        print("Set Amp gain to: {}".format(hex(self.gain)))

    def getGain(self):
        return hex(self.readByte(0x3))

    def readTemp(self):
        val1 = self.readByte(0x29)
        val2 = self.readByte(0x2a)
        temp_c = (val1 << 4) | (val2 >> 4)
        temp_c = "%0.2f" % (temp_c/16-93) 
        print(temp_c)
        return temp_c

    """ 
    Check TAS2770 operational Mode:
     - used to check if device entered in Software Shutdown
     Register:  PWR_CTL Register Address: 0x02
    """
    def checkAmpMode(self):
        pwr_ctl = self.readByte(0x02)
        print("tas2770 pwr_ctl register:{}".format(hex(pwr_ctl)))
        return (0x3 & pwr_ctl)
        