import serial
import threading

class PicSerial:
    CMD_WRITE_ID      = 'W'
    CMD_VERSION       = 'V'
    CMD_RELAY_OUT     = 'O'
    CMD_AUDIO_ANALOG  = 'A'
    CMD_AUDIO_DIGITAL = 'D'
    CMD_FULL_STATUS   = 'R'
    CMD_TEST          = 'T'

    REPLY_SUCCESS = 'OK'

    def __init__(self, port='/dev/ttyACM0'):
        self.lock = threading.Lock()
        self.serial = serial.Serial(port, 115200, timeout=1.0, write_timeout=1.0)

    def sendCommand(self, cmd):
        with self.lock:
            cmd += '\r\n'
            self.serial.reset_input_buffer()
            self.serial.write(cmd.encode('utf-8'))

            msg = ''
            while True:
                try:
                    char = self.serial.read(1).decode('utf-8')
                    if len(char) != 1:
                        break
                    if char != '\n' and char != '\r':
                        msg += char
                    elif msg and char == '\n':
                        return msg
                except Exception as e:
                    break

def pic_send_command(cmd):
    pic = PicSerial()
    res = pic.sendCommand(cmd)
    return res

def lox_set_relay(relay, enable):
    '''
    Change the state of a relay.
    @param relay 1..8
    @param enable True or False

    @raise ValueError
    '''
    if int(relay) < 1 or int(relay) > 8:
        raise ValueError('invalid relay')

    ret = pic_send_command('{}{},{}'.format(PicSerial.CMD_RELAY_OUT, relay, int(bool(enable))))
    return ret == PicSerial.REPLY_SUCCESS

def lox_read_relay(relay):
    '''
    Read the state of a relay.
    @param relay 1..8
    @return True or False

    @raise ValueError
           IOError
    '''
    if int(relay) < 1 or int(relay) > 8:
        raise ValueError('invalid relay')

    ret = pic_send_command('R')
    if not ret:
        raise IOError('no pic reply')

    allRelaysState = ret.split(',')[6]
    return bool(int(allRelaysState[8-relay]))

def lox_set_audio_analog(enable):
    '''
    Enable or disable the output interface for analog audio.
    @param enable True or False
    @return True or False
    '''
    ret = pic_send_command('{}{}'.format(PicSerial.CMD_AUDIO_ANALOG, int(bool(enable))))
    return ret == PicSerial.REPLY_SUCCESS

def lox_set_audio_digital(enable):
    '''
    Enable or disable the output interface for digital audio.
    @param enable True or False
    @return True or False
    '''
    ret = pic_send_command('{}{}'.format(PicSerial.CMD_AUDIO_DIGITAL, int(bool(enable))))
    return ret == PicSerial.REPLY_SUCCESS

