import base64
import json
import logging
import os
import uuid

from .uci import getValueOfUci

from .constants import *

log = logging.getLogger('flask-backend')

'''
Generate a new session key
Session Key is used to validate logins.
A unique ID is generated and stored in the file SESSION_KEY_FILE_PATH.
'''
def generateNewSessionKey():
    sessionKey = uuid.uuid4().hex
    try:
        f = open(SESSION_KEY_FILE_PATH, 'w')
        f.write(sessionKey)
        f.close()
        return sessionKey
    except Exception as e:
        log.error("Error while writing in file {}: {}".format(SESSION_KEY_FILE_PATH, e))
        raise e


'''
Get the session key
Session Key is used to validate logins.
Reads the file SESSION_KEY_FILE_PATH, which is where the session key is stored.
If the file doesn't exist or if it is empty, generates a new session key.
Returns the session key (string).
'''
def getSessionKey():
    try:
        # if file does not exist, create it
        if not os.path.isfile(SESSION_KEY_FILE_PATH):
            sessionKey = generateNewSessionKey()
        else: # file exists
            f = open(SESSION_KEY_FILE_PATH, 'r')
            sessionKey = f.read()
            if sessionKey == '': # file is empty
                sessionKey = generateNewSessionKey()
        return sessionKey
    except Exception as e:
        raise e


from functools import wraps
from flask import session, Response, request

'''
Check if login is required and, if so, if user is logged in.
If login is not required or if it is required and the user is already logged in, allows to proceed. Otherwise, a Non Authorized 401 HTTP Response is returned.
To check if the user is logged in, compared the sessionID from the request to the sessionID stored. If none is provided in the request or if they don't match, then login is invalid. 
'''
def login_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        try:
            login_uci = getValueOfUci('httpd', 'webserver', 'password_set')
        except Exception:
            return Response(status=500)
        else:
            try:
                if login_uci == 'true':
                    sessionId = getRequestSessionId(request.cookies)
                    if 'sessionID' not in session:
                        if sessionId is not None and os.path.isfile(SESSION_MANAGEMENT_FILE_PATH):
                            updateSessionManagementFile(sessionId, None, 'delete')
                        return Response("Login required", status=401, headers={"Content-Type": "text/plain"})
                    if session['sessionID'] != getSessionKey():
                        if sessionId is not None and os.path.isfile(SESSION_MANAGEMENT_FILE_PATH):
                            updateSessionManagementFile(sessionId, None, 'delete')
                        return Response("Login required", status=401, headers={"Content-Type": "text/plain"})
                    else:
                        return f(*args, **kwargs)
                else:
                    return f(*args, **kwargs)
            except Exception as e:
                log.error("Error while checking login: {}".format(e))
                return Response(status=500)
    return decorated_function


def getRequestSessionId(request_cookies):
    session_cookie = request_cookies.get('session')
    if session_cookie is not None:
        json_cookie = json.loads(base64.b64decode(session_cookie.split('.')[0]))
        if 'sessionID' in json_cookie:
            return json_cookie['sessionID']
    return None


def readSessionManagementFile():
    json_sessions = None
    if os.path.isfile(SESSION_MANAGEMENT_FILE_PATH):
        with open(SESSION_MANAGEMENT_FILE_PATH) as sessions_management_file:
            json_sessions = json.load(sessions_management_file)
    return json_sessions


def updateSessionManagementFile(sessionId, content, mode):
    json_sessions = readSessionManagementFile()
    if json_sessions is None:
        json_sessions = {}
    if mode == 'replace':
        json_sessions[sessionId] = content
    elif mode == 'modify':
        if sessionId not in json_sessions:
            json_sessions[sessionId] = {}
        for key, value in content.items():
            json_sessions[sessionId][key] = value
    elif mode == 'delete':
        if sessionId in json_sessions:
            del json_sessions[sessionId]
    else:
        log.error("mode unknown")
    writeSessionManagementFile(json_sessions)


def writeSessionManagementFile(json_sessions):
    with open(SESSION_MANAGEMENT_FILE_PATH, 'w') as sessions_management_file:
        sessions_management_file.write(json.dumps(json_sessions))